import torch
import torchvision
import tqdm
import clip
import numpy as np
from sklearn.linear_model import LogisticRegression
import sklearn.model_selection
from torch.utils.data import DataLoader
from torch.utils.data.sampler import SubsetRandomSampler
import os.path
import pickle
from hierarchy.hierarchy import *

def get_hierarchy(rebuild_hier=False, load_hier=True, path='resources/imagenet1k_hier.pkl'):
    hierarchy = Hierarchy()
    if rebuild_hier:
        hierarchy.build_imagenet_tree()
        hierarchy.save_to_file(path)
    if load_hier:
        hierarchy = hierarchy.load_from_file(path)
    return hierarchy

def get_clip_list():
    models_names = ['CLIP_RN50', 'CLIP_RN101', 'CLIP_RN50x4', 'CLIP_RN50x16', 'CLIP_RN50x64', 'CLIP_ViT-B~32', 'CLIP_ViT-B~16', 'CLIP_ViT-L~14', 'CLIP_ViT-L~14@336px']
    # CLIP - finetuned models
    # Following is a partial list. RN50 and ViT-B-32 is missing until their finetuning is done.
    clip_models_names = ['CLIP_finetuned_RN101', 'CLIP_finetuned_RN50x4', 'CLIP_finetuned_RN50x16', 'CLIP_finetuned_RN50x64', 'CLIP_finetuned_ViT-B~16', 'CLIP_finetuned_ViT-L~14', 'CLIP_finetuned_ViT-L~14@336px']
    # clip_models_names = ['CLIP_finetuned_RN50', 'CLIP_finetuned_RN101', 'CLIP_finetuned_RN50x4', 'CLIP_finetuned_RN50x16', 'CLIP_finetuned_RN50x64', 'CLIP_finetuned_ViT-B~32', 'CLIP_finetuned_ViT-B~16', 'CLIP_finetuned_ViT-L~14', 'CLIP_finetuned_ViT-L~14@336px']
    models_names.extend(clip_models_names)
    return models_names

imagenet_templates = [
    'itap of a {}.',
    'a bad photo of the {}.',
    'a origami {}.',
    'a photo of the large {}.',
    'the {} in a video game.',
    'art of the {}.',
    'a photo of the small {}.'
]

imagenet_templates2 = [
    'a photo of a {}.',
    'an image of a {}.'
]

hier_templates = [
    '{}, a type of {}',
    '{}, a kind of {}',
    '{}, which is a kind of {}',
    '{}, which is a type of {}',
    '{}, which is a {}',
    '{}, a {}',
    '{}, a subtype of {}',
    '{}, a member of the {} family',
    '{}, a variety of {}',
    '{}, a species of {}',
    '{}, a form of {}'
]

def zeroshot_classifier(classnames, templates, model):
    with torch.no_grad():
        zeroshot_weights = []
        for i, classname in enumerate(tqdm.tqdm(classnames)):
            texts = [template.format(classname) for template in templates]  # format with class
            if model.labels == 'hierarchical':
                parent_idx = model.hierarchy.get_parent(i)
                if parent_idx != -1:
                    parent = classnames[parent_idx]
                    hier_labels = [htemplate.format(classname,parent) for htemplate in hier_templates]
                    hier_texts = [template.format(hier_label) for template in templates for hier_label in hier_labels]
                    texts += hier_texts
                    grandparent_idx = model.hierarchy.get_parent(parent_idx)
                    if grandparent_idx != -1:
                        grandparent = classnames[grandparent_idx]
                        if grandparent == parent:
                            grandparent_idx = model.hierarchy.get_parent(grandparent_idx)
                            if grandparent_idx != -1:
                                grandparent = classnames[grandparent_idx]
                        hier_labels = [htemplate.format(classname,grandparent) for htemplate in hier_templates]
                        hier_texts = [template.format(hier_label) for template in templates for hier_label in hier_labels]
                        texts += hier_texts
            texts = clip.tokenize(texts).cuda()  # tokenize
            class_embeddings = model.clip.encode_text(texts)  # embed with text encoder
            class_embeddings /= class_embeddings.norm(dim=-1, keepdim=True)
            class_embedding = class_embeddings.mean(dim=0)
            class_embedding /= class_embedding.norm()
            zeroshot_weights.append(class_embedding)
        zeroshot_weights = torch.stack(zeroshot_weights, dim=1).cuda()
    return zeroshot_weights

class ImageNetClip(torch.nn.Module):
    def __init__(self, model, linear_probe=False, preprocess=None, imagenet_train_folder='../ImageNet/train', name='CLIP', train_batchsize=16, hierarchical=False, labels=None):
        super(ImageNetClip, self).__init__()
        self.clip = model
        self.preprocess = preprocess
        self.linear_probe = linear_probe
        self.finetuned_classifier = None
        self.hierarchical = 'hCLIP' in name
        self.labels = labels

        if linear_probe:
            # Check if a trained probe exists
            checkpoint_path = f'resources/CLIP_linear_probes/{name}.pkl'
            if os.path.exists(checkpoint_path):
                self.finetuned_classifier = pickle.load(open(checkpoint_path, 'rb'))
            else:
                # None exists, train one
                self.finetune_clip(imagenet_train_folder, train_batchsize)
                pickle.dump(self.finetuned_classifier, open(checkpoint_path, 'wb'))
        else:
            classes = self.imagenet_classes
            if self.hierarchical:
                if self.labels == 'descriptive':
                    classes = [c.lower() for c in self.imagenet_descriptive_classes]
                    # classes = self.imagenet_classes+[c[0].upper()+c[1:] for c in self.imagenet_descriptive_classes[1000:]]
                if self.labels in ['original']:
                    classes = self.imagenet_classes + self.original_imagenet_internal_nodes
                if self.labels in ['gpt', 'hierarchical']:
                    classes = self.imagenet_classes + self.gpt_imagenet_internal_nodes
                self.hierarchy = get_hierarchy(rebuild_hier=False, load_hier=True)
            self.zeroshot_classifier = zeroshot_classifier(classes, imagenet_templates, self)

    def finetune_clip(self, imagenet_train_folder, train_batchsize):
        assert self.preprocess is not None
        dataset_train = torchvision.datasets.ImageFolder(imagenet_train_folder, transform=self.preprocess)
        # Calculate the image features
        train_features, train_labels = self.get_features(dataset_train, train_batchsize=train_batchsize)

        # Perform logistic regression
        classifier = LogisticRegression(random_state=0, C=0.12, max_iter=1000, verbose=1)
        classifier.fit(train_features, train_labels)
        self.finetuned_classifier = classifier

    def get_features(self, dataset_train, train_batchsize):
        all_features = []
        all_labels = []

        # train_indices, _ = sklearn.model_selection.train_test_split(np.arange(len(dataset_train)), train_size=250000, stratify=dataset_train.targets)
        # train_loader = torch.utils.data.DataLoader(dataset_train, pin_memory=True, batch_size=train_batchsize, sampler=SubsetRandomSampler(train_indices))

        with torch.no_grad():
            # for images, labels in tqdm.tqdm(train_loader, desc='Finetuning CLIP'):
            for images, labels in tqdm.tqdm(DataLoader(dataset_train, batch_size=train_batchsize), desc='Finetuning CLIP'):
                features = self.clip.encode_image(images.cuda())

                all_features.append(features.cpu())
                all_labels.append(labels)

        return torch.cat(all_features).cpu().numpy(), torch.cat(all_labels).cpu().numpy()

    def forward(self, x):
        image_features = self.clip.encode_image(x)
        if self.linear_probe:
            logits = torch.tensor(self.finetuned_classifier.predict_proba(image_features.cpu())).cuda()  # Probably need to convert it to torch from numpy, cuda
        else:
            image_features /= image_features.norm(dim=-1, keepdim=True)
            logits = 100. * image_features @ self.zeroshot_classifier
        return logits

    imagenet_classes = ["tench", "goldfish", "great white shark", "tiger shark", "hammerhead shark", "electric ray", "stingray", "rooster", "hen", "ostrich", "brambling", "goldfinch", "house finch", "junco", "indigo bunting", "American robin", "bulbul", "jay", "magpie", "chickadee", "American dipper", "kite (bird of prey)", "bald eagle", "vulture", "great grey owl", "fire salamander", "smooth newt", "newt", "spotted salamander", "axolotl", "American bullfrog", "tree frog", "tailed frog", "loggerhead sea turtle", "leatherback sea turtle", "mud turtle", "terrapin", "box turtle", "banded gecko", "green iguana", "Carolina anole", "desert grassland whiptail lizard", "agama", "frilled-necked lizard", "alligator lizard", "Gila monster", "European green lizard", "chameleon", "Komodo dragon", "Nile crocodile", "American alligator", "triceratops", "worm snake", "ring-necked snake", "eastern hog-nosed snake", "smooth green snake", "kingsnake", "garter snake", "water snake", "vine snake", "night snake", "boa constrictor", "African rock python", "Indian cobra", "green mamba", "sea snake", "Saharan horned viper", "eastern diamondback rattlesnake", "sidewinder rattlesnake", "trilobite", "harvestman", "scorpion", "yellow garden spider", "barn spider", "European garden spider", "southern black widow", "tarantula", "wolf spider", "tick", "centipede", "black grouse", "ptarmigan", "ruffed grouse", "prairie grouse", "peafowl", "quail", "partridge", "african grey parrot", "macaw", "sulphur-crested cockatoo", "lorikeet", "coucal", "bee eater", "hornbill", "hummingbird", "jacamar", "toucan", "duck", "red-breasted merganser", "goose", "black swan", "tusker", "echidna", "platypus", "wallaby", "koala", "wombat", "jellyfish", "sea anemone", "brain coral", "flatworm", "nematode", "conch", "snail", "slug", "sea slug", "chiton", "chambered nautilus", "Dungeness crab", "rock crab", "fiddler crab", "red king crab", "American lobster", "spiny lobster", "crayfish", "hermit crab", "isopod", "white stork", "black stork", "spoonbill", "flamingo", "little blue heron", "great egret", "bittern bird", "crane bird", "limpkin", "common gallinule", "American coot", "bustard", "ruddy turnstone", "dunlin", "common redshank", "dowitcher", "oystercatcher", "pelican", "king penguin", "albatross", "grey whale", "killer whale", "dugong", "sea lion", "Chihuahua", "Japanese Chin", "Maltese", "Pekingese", "Shih Tzu", "King Charles Spaniel", "Papillon", "toy terrier", "Rhodesian Ridgeback", "Afghan Hound", "Basset Hound", "Beagle", "Bloodhound", "Bluetick Coonhound", "Black and Tan Coonhound", "Treeing Walker Coonhound", "English foxhound", "Redbone Coonhound", "borzoi", "Irish Wolfhound", "Italian Greyhound", "Whippet", "Ibizan Hound", "Norwegian Elkhound", "Otterhound", "Saluki", "Scottish Deerhound", "Weimaraner", "Staffordshire Bull Terrier", "American Staffordshire Terrier", "Bedlington Terrier", "Border Terrier", "Kerry Blue Terrier", "Irish Terrier", "Norfolk Terrier", "Norwich Terrier", "Yorkshire Terrier", "Wire Fox Terrier", "Lakeland Terrier", "Sealyham Terrier", "Airedale Terrier", "Cairn Terrier", "Australian Terrier", "Dandie Dinmont Terrier", "Boston Terrier", "Miniature Schnauzer", "Giant Schnauzer", "Standard Schnauzer", "Scottish Terrier", "Tibetan Terrier", "Australian Silky Terrier", "Soft-coated Wheaten Terrier", "West Highland White Terrier", "Lhasa Apso", "Flat-Coated Retriever", "Curly-coated Retriever", "Golden Retriever", "Labrador Retriever", "Chesapeake Bay Retriever", "German Shorthaired Pointer", "Vizsla", "English Setter", "Irish Setter", "Gordon Setter", "Brittany dog", "Clumber Spaniel", "English Springer Spaniel", "Welsh Springer Spaniel", "Cocker Spaniel", "Sussex Spaniel", "Irish Water Spaniel", "Kuvasz", "Schipperke", "Groenendael dog", "Malinois", "Briard", "Australian Kelpie", "Komondor", "Old English Sheepdog", "Shetland Sheepdog", "collie", "Border Collie", "Bouvier des Flandres dog", "Rottweiler", "German Shepherd Dog", "Dobermann", "Miniature Pinscher", "Greater Swiss Mountain Dog", "Bernese Mountain Dog", "Appenzeller Sennenhund", "Entlebucher Sennenhund", "Boxer", "Bullmastiff", "Tibetan Mastiff", "French Bulldog", "Great Dane", "St. Bernard", "husky", "Alaskan Malamute", "Siberian Husky", "Dalmatian", "Affenpinscher", "Basenji", "pug", "Leonberger", "Newfoundland dog", "Great Pyrenees dog", "Samoyed", "Pomeranian", "Chow Chow", "Keeshond", "brussels griffon", "Pembroke Welsh Corgi", "Cardigan Welsh Corgi", "Toy Poodle", "Miniature Poodle", "Standard Poodle", "Mexican hairless dog (xoloitzcuintli)", "grey wolf", "Alaskan tundra wolf", "red wolf or maned wolf", "coyote", "dingo", "dhole", "African wild dog", "hyena", "red fox", "kit fox", "Arctic fox", "grey fox", "tabby cat", "tiger cat", "Persian cat", "Siamese cat", "Egyptian Mau", "cougar", "lynx", "leopard", "snow leopard", "jaguar", "lion", "tiger", "cheetah", "brown bear", "American black bear", "polar bear", "sloth bear", "mongoose", "meerkat", "tiger beetle", "ladybug", "ground beetle", "longhorn beetle", "leaf beetle", "dung beetle", "rhinoceros beetle", "weevil", "fly", "bee", "ant", "grasshopper", "cricket insect", "stick insect", "cockroach", "praying mantis", "cicada", "leafhopper", "lacewing", "dragonfly", "damselfly", "red admiral butterfly", "ringlet butterfly", "monarch butterfly", "small white butterfly", "sulphur butterfly", "gossamer-winged butterfly", "starfish", "sea urchin", "sea cucumber", "cottontail rabbit", "hare", "Angora rabbit", "hamster", "porcupine", "fox squirrel", "marmot", "beaver", "guinea pig", "common sorrel horse", "zebra", "pig", "wild boar", "warthog", "hippopotamus", "ox", "water buffalo", "bison", "ram (adult male sheep)", "bighorn sheep", "Alpine ibex", "hartebeest", "impala (antelope)", "gazelle", "arabian camel", "llama", "weasel", "mink", "European polecat", "black-footed ferret", "otter", "skunk", "badger", "armadillo", "three-toed sloth", "orangutan", "gorilla", "chimpanzee", "gibbon", "siamang", "guenon", "patas monkey", "baboon", "macaque", "langur", "black-and-white colobus", "proboscis monkey", "marmoset", "white-headed capuchin", "howler monkey", "titi monkey", "Geoffroy's spider monkey", "common squirrel monkey", "ring-tailed lemur", "indri", "Asian elephant", "African bush elephant", "red panda", "giant panda", "snoek fish", "eel", "silver salmon", "rock beauty fish", "clownfish", "sturgeon", "gar fish", "lionfish", "pufferfish", "abacus", "abaya", "academic gown", "accordion", "acoustic guitar", "aircraft carrier", "airliner", "airship", "altar", "ambulance", "amphibious vehicle", "analog clock", "apiary", "apron", "trash can", "assault rifle", "backpack", "bakery", "balance beam", "balloon", "ballpoint pen", "Band-Aid", "banjo", "baluster / handrail", "barbell", "barber chair", "barbershop", "barn", "barometer", "barrel", "wheelbarrow", "baseball", "basketball", "bassinet", "bassoon", "swimming cap", "bath towel", "bathtub", "station wagon", "lighthouse", "beaker", "military hat (bearskin or shako)", "beer bottle", "beer glass", "bell tower", "baby bib", "tandem bicycle", "bikini", "ring binder", "binoculars", "birdhouse", "boathouse", "bobsleigh", "bolo tie", "poke bonnet", "bookcase", "bookstore", "bottle cap", "hunting bow", "bow tie", "brass memorial plaque", "bra", "breakwater", "breastplate", "broom", "bucket", "buckle", "bulletproof vest", "high-speed train", "butcher shop", "taxicab", "cauldron", "candle", "cannon", "canoe", "can opener", "cardigan", "car mirror", "carousel", "tool kit", "cardboard box / carton", "car wheel", "automated teller machine", "cassette", "cassette player", "castle", "catamaran", "CD player", "cello", "mobile phone", "chain", "chain-link fence", "chain mail", "chainsaw", "storage chest", "chiffonier", "bell or wind chime", "china cabinet", "Christmas stocking", "church", "movie theater", "cleaver", "cliff dwelling", "cloak", "clogs", "cocktail shaker", "coffee mug", "coffeemaker", "spiral or coil", "combination lock", "computer keyboard", "candy store", "container ship", "convertible", "corkscrew", "cornet", "cowboy boot", "cowboy hat", "cradle", "construction crane", "crash helmet", "crate", "infant bed", "Crock Pot", "croquet ball", "crutch", "cuirass", "dam", "desk", "desktop computer", "rotary dial telephone", "diaper", "digital clock", "digital watch", "dining table", "dishcloth", "dishwasher", "disc brake", "dock", "dog sled", "dome", "doormat", "drilling rig", "drum", "drumstick", "dumbbell", "Dutch oven", "electric fan", "electric guitar", "electric locomotive", "entertainment center", "envelope", "espresso machine", "face powder", "feather boa", "filing cabinet", "fireboat", "fire truck", "fire screen", "flagpole", "flute", "folding chair", "football helmet", "forklift", "fountain", "fountain pen", "four-poster bed", "freight car", "French horn", "frying pan", "fur coat", "garbage truck", "gas mask or respirator", "gas pump", "goblet", "go-kart", "golf ball", "golf cart", "gondola", "gong", "gown", "grand piano", "greenhouse", "radiator grille", "grocery store", "guillotine", "hair clip", "hair spray", "half-track", "hammer", "hamper", "hair dryer", "hand-held computer", "handkerchief", "hard disk drive", "harmonica", "harp", "combine harvester", "hatchet", "holster", "home theater", "honeycomb", "hook", "hoop skirt", "gymnastic horizontal bar", "horse-drawn vehicle", "hourglass", "iPod", "clothes iron", "carved pumpkin", "jeans", "jeep", "T-shirt", "jigsaw puzzle", "rickshaw", "joystick", "kimono", "knee pad", "knot", "lab coat", "ladle", "lampshade", "laptop computer", "lawn mower", "lens cap", "letter opener", "library", "lifeboat", "lighter", "limousine", "ocean liner", "lipstick", "slip-on shoe", "lotion", "music speaker", "loupe magnifying glass", "sawmill", "magnetic compass", "messenger bag", "mailbox", "tights", "one-piece bathing suit", "manhole cover", "maraca", "marimba", "mask", "matchstick", "maypole", "maze", "measuring cup", "medicine cabinet", "megalith", "microphone", "microwave oven", "military uniform", "milk can", "minibus", "miniskirt", "minivan", "missile", "mitten", "mixing bowl", "mobile home", "ford model t", "modem", "monastery", "monitor", "moped", "mortar and pestle", "graduation cap", "mosque", "mosquito net", "vespa", "mountain bike", "tent", "computer mouse", "mousetrap", "moving van", "muzzle", "metal nail", "neck brace", "necklace", "baby pacifier", "notebook computer", "obelisk", "oboe", "ocarina", "odometer", "oil filter", "pipe organ", "oscilloscope", "overskirt", "bullock cart", "oxygen mask", "product packet / packaging", "paddle", "paddle wheel", "padlock", "paintbrush", "pajamas", "palace", "pan flute", "paper towel", "parachute", "parallel bars", "park bench", "parking meter", "railroad car", "patio", "payphone", "pedestal", "pencil case", "pencil sharpener", "perfume", "Petri dish", "photocopier", "plectrum", "Pickelhaube", "picket fence", "pickup truck", "pier", "piggy bank", "pill bottle", "pillow", "ping-pong ball", "pinwheel", "pirate ship", "drink pitcher", "block plane", "planetarium", "plastic bag", "plate rack", "farm plow", "plunger", "Polaroid camera", "pole", "police van", "poncho", "pool table", "soda bottle", "plant pot", "potter's wheel", "power drill", "prayer rug", "printer", "prison", "missile", "projector", "hockey puck", "punching bag", "purse", "quill", "quilt", "race car", "racket", "radiator", "radio", "radio telescope", "rain barrel", "recreational vehicle", "fishing casting reel", "reflex camera", "refrigerator", "remote control", "restaurant", "revolver", "rifle", "rocking chair", "rotisserie", "eraser", "rugby ball", "ruler measuring stick", "sneaker", "safe", "safety pin", "salt shaker", "sandal", "sarong", "saxophone", "scabbard", "weighing scale", "school bus", "schooner", "scoreboard", "CRT monitor", "screw", "screwdriver", "seat belt", "sewing machine", "shield", "shoe store", "shoji screen / room divider", "shopping basket", "shopping cart", "shovel", "shower cap", "shower curtain", "ski", "balaclava ski mask", "sleeping bag", "slide rule", "sliding door", "slot machine", "snorkel", "snowmobile", "snowplow", "soap dispenser", "soccer ball", "sock", "solar thermal collector", "sombrero", "soup bowl", "keyboard space bar", "space heater", "space shuttle", "spatula", "motorboat", "spider web", "spindle", "sports car", "spotlight", "stage", "steam locomotive", "through arch bridge", "steel drum", "stethoscope", "scarf", "stone wall", "stopwatch", "stove", "strainer", "tram", "stretcher", "couch", "stupa", "submarine", "suit", "sundial", "sunglasses", "sunglasses", "sunscreen", "suspension bridge", "mop", "sweatshirt", "swim trunks / shorts", "swing", "electrical switch", "syringe", "table lamp", "tank", "tape player", "teapot", "teddy bear", "television", "tennis ball", "thatched roof", "front curtain", "thimble", "threshing machine", "throne", "tile roof", "toaster", "tobacco shop", "toilet seat", "torch", "totem pole", "tow truck", "toy store", "tractor", "semi-trailer truck", "tray", "trench coat", "tricycle", "trimaran", "tripod", "triumphal arch", "trolleybus", "trombone", "hot tub", "turnstile", "typewriter keyboard", "umbrella", "unicycle", "upright piano", "vacuum cleaner", "vase", "vaulted or arched ceiling", "velvet fabric", "vending machine", "vestment", "viaduct", "violin", "volleyball", "waffle iron", "wall clock", "wallet", "wardrobe", "military aircraft", "sink", "washing machine", "water bottle", "water jug", "water tower", "whiskey jug", "whistle", "hair wig", "window screen", "window shade", "Windsor tie", "wine bottle", "airplane wing", "wok", "wooden spoon", "wool", "split-rail fence", "shipwreck", "sailboat", "yurt", "website", "comic book", "crossword", "traffic or street sign", "traffic light", "dust jacket", "menu", "plate", "guacamole", "consomme", "hot pot", "trifle", "ice cream", "popsicle", "baguette", "bagel", "pretzel", "cheeseburger", "hot dog", "mashed potatoes", "cabbage", "broccoli", "cauliflower", "zucchini", "spaghetti squash", "acorn squash", "butternut squash", "cucumber", "artichoke", "bell pepper", "cardoon", "mushroom", "Granny Smith apple", "strawberry", "orange", "lemon", "fig", "pineapple", "banana", "jackfruit", "cherimoya (custard apple)", "pomegranate", "hay", "carbonara", "chocolate syrup", "dough", "meatloaf", "pizza", "pot pie", "burrito", "red wine", "espresso", "tea cup", "eggnog", "mountain", "bubble", "cliff", "coral reef", "geyser", "lakeshore", "promontory", "sandbar", "beach", "valley", "volcano", "baseball player", "bridegroom", "scuba diver", "rapeseed", "daisy", "yellow lady's slipper", "corn", "acorn", "rose hip", "horse chestnut seed", "coral fungus", "agaric", "gyromitra", "stinkhorn mushroom", "earth star fungus", "hen of the woods mushroom", "bolete", "corn cob", "toilet paper"]

    original_imagenet_internal_nodes = ['cyprinid', 'ray', 'thrush', 'corvine bird', 'newt', 'ambystomid', 'sea turtle', 'iguanid', 'agamid', 'boa', 'rattlesnake', 'coraciiform bird', 'piciform bird', 'duck', 'monotreme', 'anthozoan', 'worm', 'lobster', 'stork', 'sandpiper', 'toy spaniel', 'foxhound', 'wolfhound', 'greyhound', 'bullterrier', 'wirehair', 'pointer', 'springer spaniel', 'Belgian Sheepdog', 'sled dog', 'corgi', 'wildcat', 'viverrine', 'scarabaeid beetle', 'hymenopterous insect', 'orthopterous insect', 'dictyopterous insect', 'homopterous insect', 'odonate', 'rabbit', 'equine', 'edentate', 'lesser ape', 'lemur', 'elephant', 'procyonid', 'ganoid', 'robe', 'gown', 'guitar', 'warship', 'lighter-than-air craft', 'shed', 'protective garment', 'weight', 'handcart', 'double-reed instrument', 'towel', 'jar', 'glass', 'bicycle', 'sled', 'cap', 'memorial', 'armor plate', 'cleaning implement', 'small boat', 'opener', 'sweater', 'sailboat', 'bowed stringed instrument', 'body armor', 'cabinet', 'theater', 'knife', 'dwelling', 'shaker', 'lock', 'keyboard', 'helmet', 'staff', 'table', 'oven', 'locomotive', 'makeup', 'scarf', 'car', 'pan', 'mask', 'piano', 'basket', 'farm machine', 'sheath', 'framework', 'puzzle', 'control', 'cap', 'electro-acoustic transducer', 'bowl', 'electronic device', 'trap', 'column', 'filter', 'breathing device', 'residence', 'timer', 'camera', 'communication system', 'curtain', 'slot machine', 'reflector', 'heater', 'medical instrument', 'fabric', 'heavier-than-air craft', 'jug', 'frozen dessert', 'loaf of bread', 'sandwich', 'summer squash', 'winter squash', 'citrus', 'sauce', 'punch', 'mountain', 'shore', 'flower', 'whale', 'ridge', 'percoid fish', 'crocodilian reptile', 'elapid', 'magnifier', 'movable barrier', 'floor cover', 'gear', 'cloth covering', 'shark', 'frog', 'viper', 'phasianid', 'anseriform bird', 'marsupial', 'coelenterate', 'heron', 'schnauzer', 'setter', 'pinscher', 'poodle', 'wild dog', 'echinoderm', 'leporid', 'swine', 'antelope', 'great ape', 'clock', 'gymnastic apparatus', 'pen', 'baby bed', 'cap', 'shelter', 'swimsuit', 'optical instrument', 'necktie', 'pot', 'lamp', 'place of worship', 'brass', 'telephone', 'white goods', 'screen', 'coat', 'portable computer', 'cart', 'shoe', 'top', 'system', 'bus', 'van', 'dish', 'support', 'firearm', 'sailing vessel', 'signboard', 'armor', 'hosiery', 'bridge', 'cruciferous vegetable', 'seed', 'beating-reed instrument', 'edge tool', 'seabird', 'supporting structure', 'blind', 'dessert', 'alcohol', 'condiment', 'person', 'bird of prey', 'grouse', 'parrot', 'gastropod', 'crab', 'Sennenhunde', 'spitz', 'wolf', 'fox', 'bear', 'soft-finned fish', 'chair', 'hat', 'gun', 'wheel', 'fence', 'personal computer', 'roof', 'woodwind', 'skirt', 'aquatic mammal', 'aircraft', 'spiny-finned fish', 'source of illumination', 'squash', 'course', 'overgarment', 'bread', 'public transport', 'finch', 'salamander', 'turtle', 'shorebird', 'retriever', 'watchdog', 'domestic cat', 'bag', 'piece of cloth', 'footwear', 'stick', 'natural elevation', 'anthropoid ape', 'elasmobranch', 'spider', 'mollusk', 'big cat', 'butterfly', 'rodent', 'New World monkey', 'weapon', 'box', 'percussion instrument', 'boat', 'toiletry', 'kitchen appliance', 'ship', 'computer', 'spaniel', 'musteline mammal', 'Old World monkey', 'bottle', 'ball', 'hand tool', 'seat', 'dish', 'fungus', 'cat', 'game bird', 'cooking utensil', 'decapod crustacean', 'toy dog', 'beetle', 'shop', 'stringed instrument', 'fastener', 'truck', 'timepiece', 'amphibian', 'sports equipment', 'foodstuff', 'colubrid snake', 'arachnid', 'bovid', 'car', 'mechanical device', 'game equipment', 'crustacean', 'mercantile establishment', 'headdress', 'barrier', 'electronic equipment', 'edible fruit', 'geological formation', 'teleost fish', 'establishment', 'oscine', 'home appliance', 'restraint', 'wind instrument', 'lizard', 'nutriment', 'shepherd dog', 'measuring instrument', 'mechanism', 'vegetable', 'feline', 'monkey', 'machine', 'building', 'tool', 'even-toed ungulate', 'vessel', 'wading bird', 'fruit', 'fish', 'snake', 'sporting dog', 'ungulate', 'hound', 'vessel', 'primate', 'craft', 'motor vehicle', 'furniture', 'food', 'aquatic bird', 'garment', 'substance', 'terrier', 'protective covering', 'musical instrument', 'insect', 'instrument', 'working dog', 'self-propelled vehicle', 'plant', 'diapsid', 'equipment', 'reptile', 'implement', 'wheeled vehicle', 'arthropod', 'clothing', 'container', 'bird', 'structure', 'invertebrate', 'hunting dog', 'vehicle', 'conveyance', 'covering', 'dog', 'device', 'canine', 'carnivore', 'placental', 'mammal', 'vertebrate', 'instrumentality', 'animal', 'organism', 'artifact', 'physical entity']
    gpt_imagenet_internal_nodes = ['carp', 'ray', 'songbird', 'crow-like bird', 'salamander', 'mole salamander', 'sea turtle', 'iguanid', 'lizard-like reptile', 'boa', 'rattlesnake', 'kingfisher-like bird', 'woodpecker-like bird', 'duck', 'egg-laying mammal', 'sea anemone', 'worm', 'lobster', 'stork', 'sandpiper', 'toy spaniel', 'foxhound', 'wolfhound', 'greyhound', 'bullterrier', 'wirehair', 'pointer', 'springer spaniel', 'Belgian Sheepdog', 'sled dog', 'corgi', 'wildcat', 'civet-like mammal', 'scarab beetle', 'membrane-winged insect', 'straight-winged insect', 'net-winged insect', 'uniform-winged insect', 'toothed insect', 'rabbit', 'equine', 'toothless mammal', 'lesser ape', 'lemur', 'elephant', 'egg-laying mammal', 'scaled fish', 'robe', 'gown', 'guitar', 'warship', 'balloon-like craft', 'shed', 'protective garment', 'weight', 'handcart', 'double-piped instrument', 'towel', 'jar', 'glass', 'bicycle', 'sled', 'cap', 'memorial', 'armor plate', 'cleaning tool', 'small boat', 'opener', 'sweater', 'sailboat', 'bowed string instrument', 'body armor', 'cabinet', 'theater', 'knife', 'dwelling', 'shaker', 'lock', 'keyboard', 'helmet', 'staff', 'table', 'oven', 'locomotive', 'makeup', 'scarf', 'car', 'pan', 'mask', 'piano', 'basket', 'farm machine', 'sheath', 'framework', 'puzzle', 'control', 'cap', 'electro-acoustic transducer', 'bowl', 'electronic device', 'trap', 'column', 'filter', 'breathing device', 'residence', 'timer', 'camera', 'communication system', 'curtain', 'slot machine', 'reflector', 'heater', 'medical instrument', 'fabric', 'heavier-than-air craft', 'jug', 'frozen dessert', 'loaf of bread', 'sandwich', 'summer squash', 'winter squash', 'citrus', 'sauce', 'punch', 'mountain', 'shore', 'flower', 'whale', 'ridge', 'perch-like fish', 'crocodilian reptile', 'venomous snake', 'magnifier', 'movable barrier', 'floor cover', 'gear', 'cloth covering', 'shark', 'frog', 'viper', 'pheasant-like bird', 'waterfowl', 'marsupial', 'jellyfish-like creature', 'long-legged wading bird', 'schnauzer', 'setter', 'pinscher', 'poodle', 'wild dog', 'spiny-skinned marine invertebrate', 'rabbit or hare', 'swine', 'antelope', 'great ape', 'clock', 'gymnastic apparatus', 'pen', 'baby bed', 'cap', 'shelter', 'swimsuit', 'optical instrument', 'necktie', 'pot', 'lamp', 'place of worship', 'brass', 'telephone', 'major household appliance', 'screen', 'coat', 'laptop', 'cart', 'shoe', 'top', 'system', 'bus', 'van', 'dish', 'support', 'firearm', 'sailing vessel', 'signboard', 'armor', 'hosiery', 'bridge', 'cruciferous vegetable', 'seed', 'beating-reed instrument', 'edge tool', 'seabird', 'supporting structure', 'blind', 'dessert', 'alcohol', 'condiment', 'person', 'bird of prey', 'grouse', 'parrot', 'stomach-footed mollusk', 'crab', 'Sennenhunde', 'spitz', 'wolf', 'fox', 'bear', 'soft-finned fish', 'chair', 'hat', 'gun', 'wheel', 'fence', 'personal computer', 'roof', 'woodwind', 'skirt', 'aquatic mammal', 'aircraft', 'spiny-finned fish', 'source of illumination', 'squash', 'course', 'overgarment', 'bread', 'public transport', 'finch', 'salamander', 'turtle', 'shorebird', 'retriever', 'watchdog', 'domestic cat', 'bag', 'piece of cloth', 'footwear', 'stick', 'natural elevation', 'great ape', 'cartilaginous fish', 'spider', 'mollusk', 'big cat', 'butterfly', 'rodent', 'New World monkey', 'weapon', 'box', 'percussion instrument', 'boat', 'toiletry', 'kitchen appliance', 'ship', 'computer', 'spaniel', 'weasel-like mammal', 'Old World monkey', 'bottle', 'ball', 'hand tool', 'seat', 'dish', 'fungus', 'cat', 'game bird', 'cooking utensil', 'ten-legged sea creature with a hard shell', 'toy dog', 'beetle', 'shop', 'stringed instrument', 'fastener', 'truck', 'timepiece', 'amphibian', 'sports equipment', 'foodstuff', 'common snake', 'arachnid', 'cow-like mammal', 'car', 'mechanical device', 'game equipment', 'sea creature with a hard shell', 'mercantile establishment', 'headdress', 'barrier', 'electronic equipment', 'edible fruit', 'geological formation', 'modern bony fish', 'establishment', 'oscine', 'home appliance', 'restraint', 'wind instrument', 'lizard', 'nutriment', 'shepherd dog', 'measuring instrument', 'mechanism', 'vegetable', 'feline', 'monkey', 'machine', 'building', 'tool', 'cloven-hoofed mammal', 'vessel', 'wading bird', 'fruit', 'fish', 'snake', 'sporting dog', 'hoofed mammal', 'hound', 'vessel', 'primate', 'craft', 'motor vehicle', 'furniture', 'food', 'aquatic bird', 'garment', 'substance', 'terrier', 'protective covering', 'musical instrument', 'insect', 'instrument', 'working dog', 'self-propelled vehicle', 'plant', 'double-arched reptile', 'equipment', 'reptile', 'implement', 'wheeled vehicle', 'joint-legged critter', 'clothing', 'container', 'bird', 'structure', 'invertebrate', 'hunting dog', 'vehicle', 'conveyance', 'covering', 'dog', 'device', 'canine', 'carnivore', 'nurturing mammals', 'mammal', 'vertebrate', 'instrumentality', 'animal', 'organism', 'artifact', 'physical entity']
    # gpt_imagenet_internal_nodes = ['carp', 'ray', 'songbird', 'crow-like bird', 'salamander', 'mole salamander', 'sea turtle', 'iguanid', 'agamid', 'boa', 'rattlesnake', 'kingfisher-like bird', 'woodpecker-like bird', 'duck', 'egg-laying mammal', 'sea anemone', 'worm', 'lobster', 'stork', 'sandpiper', 'toy spaniel', 'foxhound', 'wolfhound', 'greyhound', 'bullterrier', 'wirehair', 'pointer', 'springer spaniel', 'Belgian Sheepdog', 'sled dog', 'corgi', 'wildcat', 'civet-like mammal', 'scarab beetle', 'membrane-winged insect', 'straight-winged insect', 'net-winged insect', 'uniform-winged insect', 'toothed insect', 'rabbit', 'equine', 'toothless mammal', 'lesser ape', 'lemur', 'elephant', 'egg-laying mammal', 'scaled fish', 'robe', 'gown', 'guitar', 'warship', 'balloon-like craft', 'shed', 'protective garment', 'weight', 'handcart', 'double-piped instrument', 'towel', 'jar', 'glass', 'bicycle', 'sled', 'cap', 'memorial', 'armor plate', 'cleaning tool', 'small boat', 'opener', 'sweater', 'sailboat', 'bowed string instrument', 'body armor', 'cabinet', 'theater', 'knife', 'dwelling', 'shaker', 'lock', 'keyboard', 'helmet', 'staff', 'table', 'oven', 'locomotive', 'makeup', 'scarf', 'car', 'pan', 'mask', 'piano', 'basket', 'farm machine', 'sheath', 'framework', 'puzzle', 'control', 'cap', 'electro-acoustic transducer', 'bowl', 'electronic device', 'trap', 'column', 'filter', 'breathing device', 'residence', 'timer', 'camera', 'communication system', 'curtain', 'slot machine', 'reflector', 'heater', 'medical instrument', 'fabric', 'heavier-than-air craft', 'jug', 'frozen dessert', 'loaf of bread', 'sandwich', 'summer squash', 'winter squash', 'citrus', 'sauce', 'punch', 'mountain', 'shore', 'flower', 'whale', 'ridge', 'perch-like fish', 'crocodilian reptile', 'venomous snake', 'magnifier', 'movable barrier', 'floor cover', 'gear', 'cloth covering', 'shark', 'frog', 'viper', 'pheasant-like bird', 'waterfowl', 'marsupial', 'cnidarian', 'long-legged wading bird', 'schnauzer', 'setter', 'pinscher', 'poodle', 'wild dog', 'spiny-skinned marine invertebrate', 'rabbit or hare', 'swine', 'antelope', 'great ape', 'clock', 'gymnastic apparatus', 'pen', 'baby bed', 'cap', 'shelter', 'swimsuit', 'optical instrument', 'necktie', 'pot', 'lamp', 'place of worship', 'brass', 'telephone', 'major household appliance', 'screen', 'coat', 'laptop', 'cart', 'shoe', 'top', 'system', 'bus', 'van', 'dish', 'support', 'firearm', 'sailing vessel', 'signboard', 'armor', 'hosiery', 'bridge', 'cruciferous vegetable', 'seed', 'beating-reed instrument', 'edge tool', 'seabird', 'supporting structure', 'blind', 'dessert', 'alcohol', 'condiment', 'person', 'bird of prey', 'grouse', 'parrot', 'stomach-footed mollusk', 'crab', 'Sennenhunde', 'spitz', 'wolf', 'fox', 'bear', 'soft-finned fish', 'chair', 'hat', 'gun', 'wheel', 'fence', 'personal computer', 'roof', 'woodwind', 'skirt', 'aquatic mammal', 'aircraft', 'spiny-finned fish', 'source of illumination', 'squash', 'course', 'overgarment', 'bread', 'public transport', 'finch', 'salamander', 'turtle', 'shorebird', 'retriever', 'watchdog', 'domestic cat', 'bag', 'piece of cloth', 'footwear', 'stick', 'natural elevation', 'great ape', 'cartilaginous fish', 'spider', 'mollusk', 'big cat', 'butterfly', 'rodent', 'New World monkey', 'weapon', 'box', 'percussion instrument', 'boat', 'toiletry', 'kitchen appliance', 'ship', 'computer', 'spaniel', 'weasel-like mammal', 'Old World monkey', 'bottle', 'ball', 'hand tool', 'seat', 'dish', 'fungus', 'cat', 'game bird', 'cooking utensil', 'ten-legged crustacean', 'toy dog', 'beetle', 'shop', 'stringed instrument', 'fastener', 'truck', 'timepiece', 'amphibian', 'sports equipment', 'foodstuff', 'colubrid snake', 'arachnid', 'bovid', 'car', 'mechanical device', 'game equipment', 'crustacean', 'mercantile establishment', 'headdress', 'barrier', 'electronic equipment', 'edible fruit', 'geological formation', 'modern bony fish', 'establishment', 'oscine', 'home appliance', 'restraint', 'wind instrument', 'lizard', 'nutriment', 'shepherd dog', 'measuring instrument', 'mechanism', 'vegetable', 'feline', 'monkey', 'machine', 'building', 'tool', 'cloven-hoofed mammal', 'vessel', 'wading bird', 'fruit', 'fish', 'snake', 'sporting dog', 'hoofed mammal', 'hound', 'vessel', 'primate', 'craft', 'motor vehicle', 'furniture', 'food', 'aquatic bird', 'garment', 'substance', 'terrier', 'protective covering', 'musical instrument', 'insect', 'instrument', 'working dog', 'self-propelled vehicle', 'plant', 'double-arched reptile', 'equipment', 'reptile', 'implement', 'wheeled vehicle', 'joint-legged critter', 'clothing', 'container', 'bird', 'structure', 'invertebrate', 'hunting dog', 'vehicle', 'conveyance', 'covering', 'dog', 'device', 'canine', 'carnivore', 'nurturing mammals', 'mammal', 'vertebrate', 'instrumentality', 'animal', 'organism', 'artifact', 'physical entity']    

    imagenet_descriptive_classes = ["tench, a freshwater fish with olive-green scales",
    "goldfish, a small, brightly colored fish with a distinctive fin shape",
    "great white shark, a large predatory shark with a white underbelly and pointed snout",
    "tiger shark, a powerful shark with distinctive dark stripes on its body",
    "hammerhead shark, a shark with a unique T-shaped head",
    "electric ray, a flat-bodied fish capable of producing electric shocks",
    "stingray, a flat-bodied fish with a venomous tail spine",
    "rooster, a male chicken with vibrant plumage and a distinctive comb",
    "hen, a female chicken with more subdued feathers",
    "ostrich, a large flightless bird with long legs and a distinctive neck",
    "brambling, a bird with a black head and orange breast",
    "goldfinch, a small bird with vibrant yellow and black plumage",
    "house finch, a small bird with brown and streaked plumage",
    "junco, a small bird with a dark hood and light-colored body",
    "indigo bunting, a brightly colored bird with striking blue plumage",
    "American robin, a medium-sized bird with a red breast and grayish back",
    "bulbul, a bird with a sleek body and distinctive crest on its head",
    "jay, a bird with vibrant blue feathers and a crest on its head",
    "magpie, a bird with black and white plumage and a long tail",
    "chickadee, a small bird with a distinctive black cap and white cheeks",
    "American dipper, a small bird with a dark body, short wings, and a white throat",
    "Kite, a bird of prey with a forked tail, pointed wings, and a hooked beak",
    "Bald eagle, a majestic bird with a white head, yellow beak, and strong talons",
    "Vulture, a large bird with a bald head, broad wings, and a scavenging nature",
    "Great grey owl, a big owl with a round face, grey feathers, and yellow eyes",
    "Fire salamander, a black amphibian with bright yellow spots on its body",
    "Smooth newt, a small amphibian with smooth skin and vibrant colors",
    "Newt, a small amphibian with a slender body and distinct skin markings",
    "Spotted salamander, an amphibian with dark skin and vibrant yellow spots",
    "Axolotl, an aquatic creature with feathery gills, a wide head, and a long tail",
    "American bullfrog, a large frog with a green-brown body and powerful hind legs",
    "Tree frog, a small frog with adhesive toe pads, bright colors, and a slender body",
    "Tailed frog, a frog with a distinctive tail, short body, and webbed feet",
    "Loggerhead sea turtle, a marine turtle with a large head, powerful jaws, and a reddish-brown shell",
    "Leatherback sea turtle, a massive turtle with a leathery shell and distinct ridges",
    "Mud turtle, a small turtle with a domed shell, webbed feet, and a preference for muddy habitats",
    "Terrapin, a turtle adapted to brackish water with a flattened shell and webbed feet",
    "Box turtle, a land turtle with a hinged shell that can be closed tightly",
    "Banded gecko, a small lizard with banded patterns on its body and a slender tail",
    "Green iguana, a large lizard with a green body, spines along the back, and a long tail",
    "Carolina anole, a small lizard with a slender body, dewlap, and the ability to change color",
    "Desert grassland whiptail lizard, a swift and slender lizard with a long tail and distinctive stripes",
    "Agama, a lizard with a triangular head, spiky scales, and vibrant coloration",
    "Frilled-necked lizard, a reptile with a frill around its neck, elongated body, and agile limbs",
    "Alligator lizard, a lizard with a robust body, short legs, and keeled scales",
    "Gila monster, a venomous lizard with a stocky body, patterned scales, and a beaded appearance",
    "European green lizard, a sleek lizard with green coloration, slender body, and long tail",
    "Chameleon, a lizard with independently moving eyes, prehensile tail, and the ability to change color",
    "Komodo dragon, a large lizard with powerful limbs, forked tongue, and scaly skin",
    "Nile crocodile, a large reptile with a powerful jaw, scaly armor, and a streamlined body",
    "American alligator, a robust reptile with a broad snout, armored scales, and webbed feet",
    "Triceratops, a dinosaur with a distinctive frill, three horns, and a massive body",
    "Worm snake, a small snake with a slender body and smooth scales",
    "Ring-necked snake, a snake with a ring around its neck and a slender body",
    "Eastern hog-nosed snake, a snake with an upturned snout, keeled scales, and a distinctive pattern",
    "Smooth green snake, a slender snake with smooth scales and bright green coloration",
    "Kingsnake, a snake with bold, contrasting bands and a sleek body",
    "Garter snake, a snake with longitudinal stripes and a slender body",
    "Water snake, a snake adapted for aquatic environments, with a streamlined body and scales",
    "Vine snake, a slender snake resembling a vine with a pointed head and arboreal habits",
    "Night snake, a nocturnal snake with keeled scales and a muted color palette",
    "Boa constrictor, a large snake with a muscular body and the ability to constrict prey",
    "African rock python, a massive snake with a robust body and distinctive patterning",
    "Indian cobra, a venomous snake with a hood, distinctive markings, and a slender body",
    "Green mamba, a highly venomous snake with vibrant green coloration and a slender body",
    "Sea snake, a snake adapted for marine life with a flattened tail and valve-like nostrils",
    "Saharan horned viper, a venomous snake with horns above its eyes and a desert-adapted appearance",
    "Eastern diamondback rattlesnake, a venomous snake with diamond-shaped markings and a rattle on its tail",
    "Sidewinder rattlesnake, a snake with a sidewinding motion, specialized for sandy environments",
    "Trilobite, an extinct marine arthropod with a segmented exoskeleton and distinctive body shape",
    "Harvestman, a arachnid with a small body and long, delicate legs",
    "Scorpion, an arachnid with a segmented tail, pincers, and venomous stinger",
    "Yellow garden spider, a spider with a distinctive yellow and black orb-shaped web",
    "Barn spider, a spider with a circular, intricate web and a mottled brown appearance",
    "European garden spider, a spider with a large orb web and striped legs",
    "Southern black widow, a venomous spider with a black body and red hourglass marking",
    "Tarantula, a large, hairy spider with robust legs and fangs",
    "Wolf spider, a hunting spider with a robust body and distinctive eye arrangement",
    "Tick, a small arachnid with a flattened body and mouthparts for feeding on blood",
    "Centipede, a multi-legged arthropod with a segmented body and venomous claws",
    "Black grouse, a bird with dark plumage, lyre-shaped tail feathers, and a distinctive courtship display",
    "Ptarmigan, a bird with white plumage in winter, camouflaging it in snowy environments",
    "Ruffed grouse, a bird with mottled plumage, a fan-shaped tail, and a ruff of feathers around the neck",
    "Prairie grouse, a bird adapted to grassland habitats with mottled plumage and a distinct mating dance",
    "Peafowl, a bird with iridescent plumage, an extravagant tail, and a distinctive call",
    "Quail, a small bird with a plump body, short beak, and distinctive markings",
    "Partridge, a bird with a plump body, short tail, and strong ground-dwelling habits",
    "African grey parrot, a medium-sized parrot with grey plumage and exceptional mimicry skills",
    "Macaw, a vibrant and large parrot with colorful plumage and a strong, hooked beak",
    "Sulphur-crested cockatoo, a large cockatoo with white plumage, a distinctive crest, and a raucous call",
    "lorikeet, a brightly colored parrot with a curved beak",
    "coucal, a bird with a long tail and strong beak",
    "bee eater, a slender bird with vibrant plumage and sharp beak",
    "hornbill, a large bird with a distinctive, down-curved bill",
    "hummingbird, a tiny bird with iridescent feathers and rapid wing beats",
    "jacamar, a tropical bird with a long, straight bill and colorful plumage",
    "toucan, a bird with a large, brightly colored bill",
    "duck, a waterfowl with webbed feet and a flat bill",
    "red-breasted merganser, a duck with a slender body and a distinctive red breast",
    "goose, a waterfowl with a long neck and a webbed bill",
    "black swan, a swan with black plumage and a red bill",
    "tusker, an elephant with long, curved tusks",
    "echidna, a spiny anteater with a long snout",
    "platypus, a mammal with a duck-bill and webbed feet",
    "wallaby, a small kangaroo with a slender body and long tail",
    "koala, a bear-like marsupial with fluffy ears and a large nose",
    "wombat, a stout, burrowing marsupial with a short tail",
    "jellyfish, a translucent sea creature with trailing tentacles",
    "sea anemone, a flower-like marine animal with tentacles",
    "brain coral, a hard coral with a brain-like appearance",
    "flatworm, a soft-bodied worm with a flattened shape",
    "nematode, a slender worm with a cylindrical body",
    "conch, a large sea snail with a spiral shell",
    "snail, a mollusk with a coiled shell and a slow-moving body",
    "slug, a slimy mollusk without a shell",
    "sea slug, a marine gastropod with colorful, often translucent, body",
    "chiton, a marine mollusk with overlapping plates on its shell",
    "chambered nautilus, a cephalopod with a spiral shell and multiple chambers",
    "Dungeness crab, a crab with a broad, rounded carapace",
    "rock crab, a crab with a tough exoskeleton and strong claws",
    "fiddler crab, a crab with one enlarged claw resembling a fiddle",
    "red king crab, a large crab with bright red color and spiky limbs",
    "American lobster, a lobster with large claws and a spiny carapace",
    "spiny lobster, a lobster with long antennae and a spiny carapace",
    "crayfish, a freshwater crustacean with a lobster-like appearance",
    "hermit crab, a crab with a soft abdomen that occupies empty shells",
    "isopod, a small, flattened crustacean with many legs",
    "white stork, a tall bird with a long neck and a pointed beak",
    "black stork, a large bird with black plumage and a distinctive beak",
    "spoonbill, a wading bird with a spoon-shaped bill",
    "flamingo, a tall wading bird with pink plumage and long legs",
    "little blue heron, a heron with blue-gray plumage",
    "great egret, a large heron with all-white plumage",
    "bittern bird, a secretive wading bird with a stout body",
    "crane bird, a large bird with a long neck and legs",
    "limpkin, a wading bird with a long, curved bill",
    "common gallinule, a water bird with colorful plumage and a red frontal shield",
    "American coot, a duck-like bird with dark plumage and a white bill",
    "bustard, a large, ground-dwelling bird with a stout body",
    "ruddy turnstone, a shorebird with mottled plumage and a short bill",
    "dunlin, a small sandpiper with a long, curved bill",
    "common redshank, a wader with long legs and an orange-red leg",
    "dowitcher, a shorebird with a long, straight bill",
    "oystercatcher, a bird with a long, thick bill for feeding on mollusks",
    "pelican, a large water bird with a long bill and a throat pouch",
    "king penguin, a large penguin with distinctive orange markings",
    "albatross, a large seabird with long wings and a hooked bill",
    "grey whale, a large marine mammal with mottled gray color",
    "killer whale, a powerful marine mammal with distinctive black and white markings",
    "dugong, a marine mammal with a rounded body and a tail resembling a mermaid's",
    "sea lion, a marine mammal with external ear flaps and a mane-like coat",
    "Chihuahua, a small dog breed with a tiny size and large, round eyes",
    "Japanese Chin, a toy dog breed with a distinctive pushed-in face",
    "Maltese, a toy dog breed with long, silky white hair",
    "Pekingese, a small toy dog breed with a flat face and a flowing mane",
    "Shih Tzu, a small toy dog with a flowing double coat and a pushed-in face",
    "King Charles Spaniel, a small toy spaniel with a silky coat and a distinctive flat face",
    "Papillon, a small toy spaniel with butterfly-like ears",
    "toy terrier, a tiny terrier breed with a sleek coat and alert expression",
    "Rhodesian Ridgeback, a large dog with a distinctive ridge of hair along its back",
    "Afghan Hound, a long-haired sighthound with a regal appearance",
    "Basset Hound, a short-legged hound with long ears and a wrinkled face",
    "Beagle, a small hound with a compact body and a keen sense of smell",
    "Bloodhound, a large scent hound with loose, wrinkled skin",
    "Bluetick Coonhound, a coonhound with a distinctive blue-speckled coat",
    "Black and Tan Coonhound, a coonhound with a black and tan coat",
    "Treeing Walker Coonhound, a coonhound with a treeing instinct and a smooth coat",
    "English foxhound, a medium to large hound with a sleek coat and long legs",
    "Redbone Coonhound, a coonhound with a solid red coat",
    "borzoi, a Russian sighthound with a long, silky coat and slender build",
    "Irish Wolfhound, a giant breed with a rough coat and a dignified presence",
    "Italian Greyhound, a small sighthound with a slender frame and short coat",
    "Whippet, a medium-sized sighthound with a sleek, muscular build",
    "Ibizan Hound, a slender hound with large ears and a smooth coat",
    "Norwegian Elkhound, a spitz-type dog with a dense, silver-gray coat",
    "Otterhound, a large scent hound with a shaggy coat and webbed feet",
    "Saluki, a graceful sighthound with a long, flowing coat",
    "Scottish Deerhound, a giant sighthound with a wiry coat and gentle demeanor",
    "Weimaraner, a medium to large dog with a sleek, silver-gray coat",
    "Staffordshire Bull Terrier, a muscular terrier with a distinctive head shape",
    "American Staffordshire Terrier, a strong and agile terrier with a broad head",
    "Bedlington Terrier, a terrier with a distinctive lamb-like appearance",
    "Border Terrier, a small terrier with a wiry coat and keen expression",
    "Kerry Blue Terrier, a medium-sized terrier with a curly blue-gray coat",
    "Irish Terrier, a medium-sized terrier with a wiry red coat",
    "Norfolk Terrier, a small terrier with a compact build and folded ears",
    "Norwich Terrier, a small terrier with a sturdy build and erect ears",
    "Yorkshire Terrier, a small toy terrier with a long, silky coat",
    "Wire Fox Terrier, a terrier with a wiry coat and a keen expression",
    "Lakeland Terrier, a terrier with a compact build and distinctive head carriage",
    "Sealyham Terrier, a small terrier with a distinctive white coat and folded ears",
    "Airedale Terrier, a large terrier with a wiry coat and a square head",
    "Cairn Terrier, a small terrier with a shaggy coat and erect ears",
    "Australian Terrier, a small terrier with a rough coat and a distinctive topknot",
    "Dandie Dinmont Terrier, a small terrier with a distinctive topknot and long body",
    "Boston Terrier, a small, compact dog with a tuxedo-like coat pattern",
    "Miniature Schnauzer, a small terrier with a wiry beard and bushy eyebrows",
    "Giant Schnauzer, a large, powerful terrier with a dense, wiry coat",
    "Standard Schnauzer, a medium-sized terrier with a salt-and-pepper coat",
    "Scottish Terrier, a small terrier with a distinctive wiry coat and pointy ears",
    "Tibetan Terrier, a medium-sized dog with a long, flowing coat and bushy tail",
    "Australian Silky Terrier, a small terrier with a silky blue and tan coat",
    "Soft-coated Wheaten Terrier, a medium-sized terrier with a soft, wheaten-colored coat",
    "West Highland White Terrier, a small terrier with a crisp white coat",
    "Lhasa Apso, a small dog with a long, flowing coat and a distinctive mane",
    "Flat-Coated Retriever, a medium to large retriever with a sleek black coat",
    "Curly-coated Retriever, a retriever with tight, water-resistant curls",
    "Golden Retriever, a large retriever with a dense, golden-colored coat",
    "Labrador Retriever, a medium to large retriever with a short, dense coat",
    "Chesapeake Bay Retriever, a retriever with a dense, waterproof coat and webbed feet",
    "German Shorthaired Pointer, a versatile pointer with a short, dense coat",
    "Vizsla, a medium-sized pointer with a short, smooth coat and a golden rust color",
    "English Setter, a medium-sized setter with a silky coat and feathered tail",
    "Irish Setter, a setter with a sleek, mahogany-red coat",
    "Gordon Setter, a setter with a dense, black-and-tan coat",
    "Brittany dog, a medium-sized spaniel with a dense, orange and white coat",
    "Clumber Spaniel, a large spaniel with a dense, white coat and orange markings",
    "English Springer Spaniel, a medium-sized spaniel with a distinctive coat pattern",
    "Welsh Springer Spaniel, a medium-sized spaniel with a rich red and white coat",
    "Cocker Spaniel, a small spaniel with a silky coat and long ears",
    "Sussex Spaniel, a spaniel with a dense, golden liver-colored coat",
    "Irish Water Spaniel, a spaniel with a curly, water-resistant coat and a topknot",
    "Kuvasz, a large, white dog with a dense double coat",
    "Schipperke, a small, black dog with a dense, double coat and erect ears",
    "Groenendael dog, a black Belgian Shepherd with a long, flowing coat",
    "Malinois, a Belgian Shepherd with a short, fawn-colored coat",
    "Briard, a large herding dog with a long, wavy coat",
    "Australian Kelpie, a medium-sized herding dog with a short coat and alert expression",
    "Komondor, a large, white dog with distinctive corded coat",
    "Old English Sheepdog, a large herding dog with a shaggy, gray and white coat",
    "Shetland Sheepdog, a small herding dog with a dense, sable and white coat",
    "collie, a medium to large herding dog with a luxurious coat and distinctive mane",
    "Border Collie, a medium-sized herding dog with a sleek coat and intense gaze",
    "Bouvier des Flandres dog, a large herding dog with a rough, dense coat",
    "Rottweiler, a large, powerful dog with a black and tan coat",
    "German Shepherd Dog, a large herding dog with a strong, athletic build",
    "Dobermann, a sleek, medium to large dog with a short, black and tan coat",
    "Miniature Pinscher, a small dog with a sleek coat and erect ears",
    "Greater Swiss Mountain Dog, a large, tri-colored dog with a dense, short coat",
    "Bernese Mountain Dog, a large, tricolored dog with a thick, long coat",
    "Appenzeller Sennenhund, a medium-sized, tricolored dog with a short coat",
    "Entlebucher Sennenhund, a small, tricolored dog with a dense, short coat",
    "Boxer, a medium to large dog with a short coat and a strong build",
    "Bullmastiff, a large, muscular dog with a short coat and a wrinkled face",
    "Tibetan Mastiff, a large, powerful dog with a dense, mane-like coat",
    "French Bulldog, a small, compact dog with a smooth coat and bat-like ears",
    "Great Dane, a giant dog with a sleek coat and a tall, muscular build",
    "St Bernard, a giant dog with a dense, long coat and a distinctive face mask",
    "husky, a medium to large dog with a dense coat, erect ears, and distinctive markings",
    "Alaskan Malamute, a large, powerful dog with a dense coat and erect ears",
    "Siberian Husky, a medium-sized dog with a thick double coat, erect ears, and distinctive markings",
    "Dalmatian, a medium-sized dog with a distinctive black or liver-spotted coat",
    "Affenpinscher, a small toy dog with a wiry coat and a distinctive monkey-like face",
    "Basenji, a medium-sized hound with a sleek coat and tightly curled tail",
    "pug, a small dog with a wrinkled face, curled tail, and compact build",
    "Leonberger, a giant dog with a lion-like mane and a thick, double coat",
    "Newfoundland dog, a large working dog with a water-resistant coat and webbed feet",
    "Great Pyrenees dog, a giant dog with a thick, white coat and a majestic presence",
    "Samoyed, a fluffy, white dog with a smiling expression and erect ears",
    "Pomeranian, a small dog with a fluffy double coat and fox-like face",
    "Chow Chow, a distinctive-looking dog with a lion-like mane and a blue-black tongue",
    "Keeshond, a medium-sized dog with a plush, thick coat and distinctive facial markings",
    "brussels griffon, a small toy dog with a wiry coat and expressive face",
    "Pembroke Welsh Corgi, a small herding dog with a foxy face and short legs",
    "Cardigan Welsh Corgi, a small herding dog with a long body, short legs, and bushy tail",
    "Toy Poodle, a small poodle with a curly coat and elegant appearance",
    "Miniature Poodle, a medium-sized poodle with a curly coat and graceful build",
    "Standard Poodle, a large poodle with a curly coat and athletic physique",
    "Mexican hairless dog (xoloitzcuintli), a medium-sized dog with a hairless coat and a sleek build",
    "grey wolf, a large wild canine with a gray coat and strong build",
    "Alaskan tundra wolf, a gray wolf adapted to the Arctic tundra environment",
    "red wolf or maned wolf, a tall, long-legged wolf with a reddish coat and distinctive mane",
    "coyote, a medium-sized wild canine with a grayish-brown coat",
    "dingo, a wild dog native to Australia with a yellow-brown coat",
    "dhole, a wild dog with a reddish-brown coat and distinctive white markings",
    "African wild dog, a social wild dog with a colorful coat pattern",
    "hyena, a carnivorous mammal with a robust build and distinctive laugh-like vocalizations",
    "red fox, a small carnivorous mammal with a bushy tail and orange-red fur",
    "kit fox, a small fox with a grayish-brown coat and large ears",
    "Arctic fox, a small fox adapted to cold environments with a thick white coat",
    "grey fox, a small fox with a grayish coat and a black-tipped tail",
    "tabby cat, a domestic cat with a distinctive striped or spotted coat pattern",
    "tiger cat, a domestic cat with a coat pattern resembling that of a tiger",
    "Persian cat, a long-haired cat breed with a flat face and luxurious coat",
    "Siamese cat, a sleek cat breed with a short coat and striking blue almond-shaped eyes",
    "Egyptian Mau, a spotted cat breed with a short coat and distinctive markings",
    "cougar, a large wild cat with a tan coat and a powerful build",
    "lynx, a medium-sized wild cat with tufted ears and a short tail",
    "leopard, a large wild cat with a spotted coat and powerful build",
    "snow leopard, a high-altitude wild cat with a thick, spotted coat",
    "jaguar, a large wild cat with a powerful build and a distinctive rosette pattern",
    "lion, a large wild cat with a golden mane (in males) and a powerful build",
    "tiger, a large wild cat with a striped coat and a powerful build",
    "cheetah, a slender, swift wild cat with distinctive black tear marks",
    "brown bear, a large bear with a shaggy brown coat and a powerful build",
    "American black bear, a medium-sized bear with a sleek black coat",
    "polar bear, a large bear adapted to Arctic environments with a thick white coat",
    "sloth bear, a medium-sized bear with a shaggy coat and a distinct facial appearance",
    "mongoose, a small carnivorous mammal with a sleek body and a pointed snout",
    "meerkat, a small mammal with a slender body and a distinctive upright posture",
    "tiger beetle, a predatory beetle with large mandibles and a quick, agile movement",
    "ladybug, a small beetle with a rounded, spotted shell",
    "ground beetle, a beetle with a robust body and a strong mandible for hunting",
    "longhorn beetle, a beetle with long antennae and distinctive long, segmented horns",
    "leaf beetle, a small beetle with a flattened body and often colorful markings",
    "dung beetle, a beetle with a strong, compact body and a habit of rolling dung",
    "rhinoceros beetle, a large beetle with a horn-like structure on its head",
    "weevil, a small beetle with a distinctive long snout",
    "fly, a small insect with a single pair of wings and a rapid flight",
    "bee, a flying insect known for its pollination role and striped appearance",
    "ant, a social insect with a slender waist and often a stinging capability",
    "grasshopper, an herbivorous insect with strong hind legs for jumping",
    "cricket insect, an insect known for its chirping sound, often associated with nighttime",
    "stick insect, an insect with a body resembling a twig for camouflage",
    "cockroach, an insect with a flat, oval body and rapid, scurrying movement",
    "praying mantis, a predatory insect with a long body and front legs adapted for grasping",
    "cicada, an insect known for its loud, buzzing sound",
    "leafhopper, a small, plant-feeding insect with a broad, flattened body",
    "lacewing, an insect with delicate wings and predatory larvae",
    "dragonfly, a flying insect with large transparent wings and a slender body",
    "damselfly, a delicate flying insect with slender bodies and folded wings",
    "red admiral butterfly, a butterfly with distinctive red bands on black wings",
    "ringlet butterfly, a butterfly with brown wings and distinctive eye spots",
    "monarch butterfly, a large butterfly with orange wings and black veins",
    "small white butterfly, a small butterfly with mostly white wings",
    "sulphur butterfly, a butterfly with bright yellow wings",
    "gossamer-winged butterfly, a delicate butterfly with translucent wings",
    "starfish, a marine animal with multiple arms radiating from a central disk",
    "sea urchin, a spiny marine creature with a spherical body",
    "sea cucumber, a marine invertebrate with a leathery skin and elongated body",
    "cottontail rabbit, a small rabbit with a fluffy, cotton-like tail",
    "hare, a larger rabbit-like mammal with long ears and powerful hind legs",
    "Angora rabbit, a domestic rabbit breed with long, silky fur",
    "hamster, a small rodent with a stocky body and cheek pouches for storing food",
    "porcupine, a rodent covered in sharp spines or quills",
    "fox squirrel, a tree-dwelling rodent with a bushy tail and reddish-brown fur",
    "marmot, a large ground-dwelling rodent with a stout body and short legs",
    "beaver, a large rodent with webbed feet and a flattened tail",
    "guinea pig, a small domesticated rodent with a compact body and no tail",
    "common sorrel horse, a domesticated horse with a reddish-brown coat",
    "zebra, a wild horse-like mammal with distinctive black and white stripes",
    "pig, a domesticated mammal with a stout body and a snout",
    "wild boar, a wild pig with tusks and a bristly coat",
    "warthog, a wild pig with large tusks and wart-like facial protuberances",
    "hippopotamus, a large aquatic mammal with a barrel-shaped body and large mouth",
    "ox, a domesticated bovine used for plowing and other labor",
    "water buffalo, a domesticated bovine with large curved horns",
    "bison, a large, shaggy-haired mammal with a hump on its back",
    "ram (adult male sheep), a male sheep with curved horns",
    "bighorn sheep, a wild sheep with large, curved horns",
    "Alpine ibex, a wild goat with long, curved horns",
    "hartebeest, a large African antelope with a sloping back",
    "impala (antelope), a medium-sized African antelope with lyre-shaped horns",
    "gazelle, a slender, graceful antelope with curved horns",
    "arabian camel, a large mammal with a humped back, adapted for desert life",
    "llama, a domesticated South American mammal related to the camel",
    "weasel, a small carnivorous mammal with a slender body and short legs",
    "mink, a semi-aquatic mammal with a sleek fur coat",
    "European polecat, a small carnivorous mammal with a dark mask-like face",
    "black-footed ferret, a North American carnivore with a black mask and feet",
    "otter, a semi-aquatic mammal with a streamlined body and webbed feet",
    "skunk, a small mammal with a distinctive black-and-white color pattern",
    "badger, a burrowing mammal with a stocky body and distinctive facial markings",
    "armadillo, a mammal with a bony armored shell covering its body",
    "three-toed sloth, a slow-moving mammal with a unique three-toed foot structure",
    "orangutan, a large, arboreal ape with a shaggy coat and distinctive face",
    "gorilla, a large, powerful ape with a stocky build and dark fur",
    "chimpanzee, a small, intelligent ape with a human-like appearance",
    "gibbon, a small, arboreal ape with long arms and a slender build",
    "siamang, a large, black-furred gibbon with a distinctive throat sac",
    "guenon, a small to medium-sized monkey with a slender body and long tail",
    "patas monkey, a medium-sized monkey with long legs and a reddish-brown coat",
    "baboon, a large monkey with a dog-like face and a long, naked tail",
    "macaque, a medium-sized monkey with a stout build and a short tail",
    "langur, a long-tailed monkey with a slender build and distinctive fur",
    "black-and-white colobus, a large monkey with a black-and-white coat and long tail",
    "proboscis monkey, a monkey with a long, protruding nose and potbelly",
    "marmoset, a small monkey with a tiny body and tufted ears",
    "white-headed capuchin, a medium-sized monkey with a cap-like arrangement of fur on its head",
    "howler monkey, a large monkey with a prehensile tail and a loud vocalization",
    "titi monkey, a small monkey with a long, non-prehensile tail",
    "Geoffroy's spider monkey, a large monkey with a slender build and prehensile tail",
    "common squirrel monkey, a small monkey with a long tail and large eyes",
    "ring-tailed lemur, a medium-sized primate with a distinctive striped tail",
    "indri, a large lemur with a fluffy, black-and-white coat",
    "Asian elephant, a large, intelligent mammal with a long trunk and large ears",
    "African bush elephant, a larger elephant species with larger ears",
    "red panda, a small, arboreal mammal with a reddish-brown coat",
    "giant panda, a large bear-like mammal with distinctive black and white markings",
    "snoek fish, a slender, elongated fish with a pointed snout and silver scales",
    "eel, a long, snake-like fish with a smooth, slimy skin",
    "silver salmon, a salmon species with a silvery body and distinctive humpback shape",
    "rock beauty fish, a colorful marine fish with vibrant patterns and a compressed body",
    "clownfish, a small, brightly colored fish with distinctive stripes and anemone association",
    "sturgeon, a large, primitive fish with bony plates and a long snout",
    "gar fish, a predatory fish with a long, slender body and needle-like teeth",
    "lionfish, a venomous marine fish with flamboyant fins and stripes",
    "pufferfish, a fish that can inflate itself into a ball as a defense mechanism",
    "abacus, a counting device with beads on rods",
    "abaya, a loose-fitting, black robe-like dress worn by some Muslim women",
    "academic gown, a traditional robe worn during academic ceremonies",
    "accordion, a musical instrument with a bellows and keys",
    "acoustic guitar, a stringed instrument with a hollow body and sound hole",
    "aircraft carrier, a large naval ship designed for carrying and launching aircraft",
    "airliner, a large passenger aircraft for commercial flights",
    "airship, a lighter-than-air craft with a gas-filled envelope",
    "altar, a raised platform used for religious rituals",
    "ambulance, a vehicle equipped for transporting the sick or injured",
    "amphibious vehicle, a vehicle capable of operating on land and water",
    "analog clock, a timekeeping device with hands and a face",
    "apiary, a place where bees are kept for the production of honey",
    "apron, a protective garment worn over the front of the body and tied around the back",
    "trash can, a container for holding waste or garbage",
    "assault rifle, a lightweight rifle designed for rapid fire",
    "backpack, a bag carried on the back, often used for hiking or school",
    "bakery, a place where bread and other baked goods are made and sold",
    "balance beam, a gymnastics apparatus used for balancing routines",
    "balloon, a flexible bag filled with gas or air and used for decoration or travel",
    "ballpoint pen, a pen with a small rotating ball at the tip for ink application",
    "Band-Aid, a brand of adhesive bandage for covering minor wounds",
    "banjo, a stringed musical instrument with a round body and a long neck",
    "baluster/handrail, a vertical support or railing, often used on stairs or balconies",
    "barbell, a weightlifting equipment with a long bar and weights on each end",
    "barber chair, a chair used by barbers for cutting and styling hair",
    "barbershop, a place where haircuts and grooming services are provided",
    "barn, a large building used for storing hay, grain, or housing livestock",
    "barometer, a device for measuring atmospheric pressure",
    "barrel, a cylindrical container with a flat top and bottom",
    "wheelbarrow, a small, one-wheeled cart for carrying loads",
    "baseball, a sport played with a bat and ball",
    "basketball, a sport played with a ball and a hoop",
    "bassinet, a small bed for a baby",
    "bassoon, a double-reed woodwind instrument",
    "swimming cap, a tight-fitting cap worn by swimmers",
    "bath towel, a large towel used for drying after bathing",
    "bathtub, a container for holding water for bathing",
    "station wagon, a car with a longer roofline and a rear cargo area",
    "lighthouse, a tall tower with a light to guide ships at sea",
    "beaker, a cylindrical container used for holding liquids",
    "military hat (bearskin or shako), a distinctive hat worn by military personnel",
    "beer bottle, a glass or plastic container for holding beer",
    "beer glass, a vessel for serving beer, typically made of glass",
    "bell tower, a tower containing one or more bells",
    "baby bib, a garment worn by infants to protect their clothing while eating",
    "tandem bicycle, a bicycle built for two riders",
    "bikini, a two-piece swimsuit for women",
    "ring binder, a type of file binder with rings for holding punched papers",
    "binoculars, optical instruments with two telescopes for viewing distant objects",
    "birdhouse, a small structure for birds to nest in",
    "boathouse, a building used for storing boats",
    "bobsleigh, a vehicle used in the sport of bobsledding",
    "bolo tie, a type of necktie made of a cord or braided leather with decorative metal tips",
    "poke bonnet, a women's bonnet with a projecting brim",
    "bookcase, a piece of furniture with shelves for holding books",
    "bookstore, a shop where books are sold",
    "bottle cap, a metal or plastic cap used to seal a bottle",
    "hunting bow, a weapon used for archery and hunting",
    "bow tie, a type of necktie that forms a bow",
    "brass memorial plaque, a commemorative plaque made of brass",
    "bra, a women's undergarment for supporting the breasts",
    "breakwater, a barrier built to protect a shoreline from the force of waves",
    "breastplate, a piece of armor covering the chest",
    "broom, a cleaning tool with a bundle of bristles attached to a handle",
    "bucket, a cylindrical container with a handle for carrying liquids or materials",
    "buckle, a fastening device with a frame and a prong",
    "bulletproof vest, a garment designed to protect against gunfire",
    "high-speed train, a fast passenger train",
    "butcher shop, a store where meat is sold",
    "taxicab, a vehicle for hire with a driver, used to transport passengers",
    "cauldron, a large metal pot for cooking over an open fire",
    "candle, a cylindrical block of wax with a wick, burned for light",
    "cannon, a large-caliber gun used for firing projectiles",
    "canoe, a narrow, open watercraft propelled by paddles",
    "can opener, a device for opening sealed cans",
    "cardigan, a knitted sweater with an open front",
    "car mirror, a reflective surface on a vehicle for rearview visibility",
    "carousel, a rotating platform with seats for amusement rides",
    "tool kit, a set of tools for various tasks",
    "cardboard box/carton, a container made of cardboard",
    "car wheel, a circular device on a vehicle that rotates to move the vehicle",
    "automated teller machine, a machine for banking transactions",
    "cassette, a small plastic case containing magnetic tape for audio recording",
    "cassette player, a device for playing audio cassettes",
    "castle, a large fortified building with towers and walls",
    "catamaran, a boat with two parallel hulls",
    "CD player, a device for playing compact discs",
    "cello, a stringed musical instrument",
    "mobile phone, a handheld device for telecommunications",
    "chain, a series of connected links",
    "chain-link fence, a fence made of interlocking steel links",
    "chain mail, armor made of interlocking metal rings",
    "chainsaw, a mechanical saw with a rotating chain for cutting wood",
    "storage chest, a large container for storing items",
    "chiffonier, a tall, narrow chest of drawers",
    "bell or wind chime, a hanging object that produces sound when moved by the wind",
    "China cabinet, a tall wooden furniture piece with glass doors and shelves",
    "Christmas stocking, a hanging sock-shaped bag adorned with festive colors",
    "Church, a large building with a steeple, windows, and a cross on top",
    "Movie theater, a dark room with rows of seats facing a big screen",
    "Cleaver, a heavy knife with a rectangular blade and a wooden handle",
    "Cliff dwelling, a structure built into the side of a cliff or rock",
    "Cloak, a loose-fitting garment worn over clothes",
    "Clogs, sturdy shoes with a thick wooden sole",
    "Cocktail shaker, a metal container for mixing and serving drinks",
    "Coffee mug, a cylindrical container with a handle for holding hot beverages",
    "Coffeemaker, a machine with a carafe, filter, and heating element for brewing coffee",
    "Spiral or coil, a curved or twisted form",
    "Combination lock, a security device with dials or numbers for access",
    "Computer keyboard, a flat device with keys for inputting data",
    "Candy store, a shop filled with various sweets and treats",
    "Container ship, a large vessel designed for transporting goods in containers",
    "Convertible, a car with a retractable roof",
    "Corkscrew, a tool for removing corks from bottles",
    "Cornet, a brass musical instrument with a conical shape",
    "Cowboy boot, a high-heeled, pointed-toe leather boot",
    "Cowboy hat, a wide-brimmed hat often associated with Western wear",
    "Cradle, a small bed for an infant that rocks back and forth",
    "Construction crane, a tall machine with a horizontal arm for lifting heavy objects",
    "Crash helmet, a protective headgear worn by motorcycle riders",
    "Crate, a sturdy wooden or plastic container for storage or transport",
    "Infant bed, a small bed designed for babies",
    "Crock Pot, a slow cooker with a ceramic pot for cooking",
    "Croquet ball, a colored wooden or plastic ball used in the game of croquet",
    "Crutch, a support device used by people with difficulty walking",
    "Cuirass, a piece of armor that covers the chest and back",
    "Dam, a barrier built across a river to control water flow",
    "Desk, a piece of furniture with a flat surface for writing or working",
    "Desktop computer, a personal computer designed to sit on a desk",
    "Rotary dial telephone, a telephone with a rotating dial for number input",
    "Diaper, a cloth or disposable garment worn by infants",
    "Digital clock, a timekeeping device with numerical displays",
    "Digital watch, a wristwatch with a digital display",
    "Dining table, a flat surface with legs for serving meals",
    "Dishcloth, a small cloth used for washing dishes",
    "Dishwasher, a machine for cleaning dishes automatically",
    "Disc brake, a type of braking system with a flat disc",
    "Dock, a platform for loading and unloading ships",
    "Dog sled, a sled pulled by dogs, often used in cold climates",
    "Dome, a rounded roof or ceiling structure",
    "Doormat, a mat placed at the entrance for wiping shoes",
    "Drilling rig, a machine for drilling holes in the ground",
    "Drum, a percussion instrument with a hollow body and a skin stretched over one or both ends",
    "Drumstick, a slender stick used for playing drums",
    "Dumbbell, a short bar with weights at each end",
    "Dutch oven, a heavy cooking pot with thick walls and a tight-fitting lid",
    "Electric fan, a device with rotating blades for circulating air",
    "Electric guitar, a stringed musical instrument with an amplified sound",
    "Electric locomotive, a powered train engine with electric propulsion",
    "Entertainment center, a furniture piece designed to hold electronic equipment",
    "Envelope, a flat paper container for mailing letters",
    "Espresso machine, a device for making concentrated coffee",
    "Face powder, a cosmetic powder for facial application",
    "Feather boa, a long decorative accessory made of feathers",
    "Filing cabinet, a piece of office furniture with drawers for organizing documents",
    "Fireboat, a specialized boat equipped for firefighting on water",
    "Fire truck, a vehicle with equipment for firefighting and rescue",
    "Fire screen, a decorative barrier used to shield from sparks",
    "Flagpole, a tall pole for flying flags",
    "Flute, a musical instrument with a slender tube and finger holes",
    "Folding chair, a portable chair with a hinged frame",
    "Football helmet, a protective headgear worn in American football",
    "Forklift, a powered industrial truck for lifting and moving heavy objects",
    "Fountain, a decorative structure with water flowing from a central source",
    "Fountain pen, a pen with a refillable ink reservoir",
    "Four-poster bed, a bed with tall posts at each corner",
    "Freight car, a rail vehicle for transporting goods",
    "French horn, a brass musical instrument with a coiled tube",
    "Frying pan, a shallow cooking pan with a long handle",
    "Fur coat, a garment made from animal fur",
    "Garbage truck, a vehicle designed for collecting and transporting waste",
    "Gas mask or respirator, a protective device for filtering air",
    "Gas pump, a machine for dispensing fuel",
    "Goblet, a stemmed drinking glass with a wide bowl",
    "Go-kart, a small motorized vehicle for recreational racing",
    "Golf ball, a small, dimpled ball used in the sport of golf",
    "Golf cart, a small vehicle for transporting golfers and their equipment",
    "Gondola, a narrow boat used in Venetian canals",
    "Gong, a large metal percussion instrument",
    "Gown, a long, formal dress",
    "Grand piano, a large, horizontal piano with a soundboard and strings",
    "Greenhouse, a structure for cultivating plants in a controlled environment",
    "Radiator grille, a decorative grill on the front of a vehicle",
    "Grocery store, a retail store selling food and household items",
    "Guillotine, a device for carrying out executions by beheading",
    "Hair clip, a small accessory for holding hair in place",
    "Hair spray, a cosmetic product for styling hair",
    "Half-track, a vehicle with tracks in the rear and wheels in the front",
    "Hammer, a tool with a heavy head for driving nails or breaking objects",
    "Hamper, a basket for holding and carrying items",
    "Hair dryer, an electrical device for drying hair",
    "Hand-held computer, a portable electronic device for computing",
    "Handkerchief, a small cloth used for personal hygiene or decoration",
    "Hard disk drive, a data storage device for computers",
    "Harmonica, a small wind instrument with metal reeds",
    "Harp, a large stringed musical instrument with a triangular frame",
    "Combine harvester, a machine for harvesting crops",
    "Hatchet, a small, short-handled axe",
    "Holster, a holder for a firearm, worn on a belt or strap",
    "Home theater, a multimedia system for home entertainment",
    "Honeycomb, a structure of hexagonal cells made by bees for honey storage",
    "Hook, a curved device for hanging or holding items",
    "Hoop skirt, a structured undergarment to support a wide skirt",
    "Gymnastic horizontal bar, a raised bar for gymnastic exercises",
    "Horse-drawn vehicle, a vehicle pulled by horses",
    "Hourglass, a timekeeping device with sand flowing between two glass bulbs",
    "iPod, a portable digital music player",
    "Clothes iron, a device for smoothing wrinkles in fabric",
    "Carved pumpkin, a pumpkin with a face or design cut into it",
    "Jeans, a type of durable, casual pants made of denim",
    "Jeep, a rugged, off-road vehicle",
    "T-shirt, a short-sleeved casual shirt",
    "Jigsaw puzzle, a puzzle made of small, interlocking pieces",
    "Rickshaw, a two-wheeled vehicle pulled by a person",
    "Joystick, a control device for video games or aircraft",
    "Kimono, a traditional Japanese garment",
    "Knee pad, a protective pad worn over the knee",
    "Knot, a fastening made by tying a flexible cord",
    "Lab coat, a white coat worn by medical professionals or scientists",
    "Ladle, a long-handled spoon with a deep bowl for serving liquids",
    "Lampshade, a covering for a lamp to diffuse or direct light",
    "Laptop computer, a portable personal computer",
    "Lawn mower, a machine for cutting grass",
    "Lens cap, a protective cover for a camera lens",
    "Letter opener, a tool for opening sealed envelopes",
    "Library, a place where books and other materials are stored and accessed",
    "Lifeboat, a small boat carried on a ship for emergency use",
    "Lighter, a device for creating a flame for lighting cigarettes or candles",
    "Limousine, a luxurious, chauffeur-driven car",
    "Ocean liner, a large passenger ship for long sea voyages",
    "Lipstick, a cosmetic product for coloring the lips",
    "Slip-on shoe, a shoe without laces that can be easily put on",
    "Lotion, a liquid or cream applied to the skin for moisturizing",
    "Music speaker, a device for playing audio sound",
    "Loupe magnifying glass, a small magnifying lens for close inspection",
    "Sawmill, a facility for processing logs into lumber",
    "Magnetic compass, a navigational tool indicating direction",
    "Messenger bag, a shoulder bag for carrying items",
    "Mailbox, a container for receiving and collecting mail",
    "Tights, close-fitting hosiery covering the legs and feet",
    "One-piece bathing suit, a single-piece swimsuit",
    "Manhole cover, a heavy metal lid covering a manhole",
    "Maraca, a percussion instrument filled with seeds or beans",
    "Marimba, a musical instrument with wooden bars struck by mallets",
    "Mask, a covering for the face or head",
    "Matchstick, a small wooden stick with a flammable tip for starting fires",
    "Maypole, a tall pole decorated with ribbons, traditionally danced around in May",
    "Maze, a complex network of paths or passages",
    "Measuring cup, a container with marked units for measuring liquids or solids",
    "Medicine cabinet, a storage cabinet for medical supplies",
    "Megalith, a large, prehistoric stone monument",
    "Microphone, a device for converting sound into an electrical signal",
    "Microwave oven, an appliance for cooking or reheating food using microwaves",
    "Military uniform, distinctive clothing worn by military personnel",
    "Milk can, a container for transporting and storing milk",
    "Minibus, a small passenger bus",
    "Miniskirt, a short skirt with a hemline well above the knee",
    "Minivan, a small van for transporting passengers",
    "Missile, a guided projectile designed for flight",
    "Mitten, a hand covering with a separate section for the thumb",
    "Mixing bowl, a deep bowl for combining ingredients in cooking",
    "Mobile home, a portable dwelling on wheels",
    "Ford Model T, an early mass-produced automobile",
    "Modem, a device for modulating and demodulating data for computer communication",
    "Monastery, a large religious building with spires, arches, and courtyards",
    "Monitor, a rectangular screen with buttons and a stand",
    "Moped, a small motorized scooter with two wheels",
    "Mortar and pestle, a pair of stone tools for grinding substances",
    "Graduation cap, a square-shaped hat with a tassel",
    "Mosque, a domed building with minarets and intricate designs",
    "Mosquito net, a fine mesh fabric hung to protect from insects",
    "Vespa, a sleek, compact scooter with two wheels",
    "Mountain bike, a sturdy bicycle with thick tires and multiple gears",
    "Tent, a portable shelter with fabric walls and poles",
    "Computer mouse, a small device with buttons and a scroll wheel",
    "Mousetrap, a contraption with a spring for catching rodents",
    "Moving van, a large vehicle for transporting goods",
    "Muzzle, a device covering an animal's mouth, often with straps",
    "Metal nail, a slender pointed pin for fastening objects",
    "Neck brace, a supportive device worn around the neck",
    "Necklace, a string of decorative beads or stones",
    "Baby pacifier, a small nipple-shaped device for infants",
    "Notebook computer, a portable device with a hinged screen and keyboard",
    "Obelisk, a tall, four-sided pillar with a pointed top",
    "Oboe, a woodwind instrument with a slender, curved shape",
    "Ocarina, a small, rounded musical instrument with finger holes",
    "Odometer, a circular gauge for measuring distance traveled",
    "Oil filter, a cylindrical device for cleaning engine oil",
    "Pipe organ, a large musical instrument with multiple pipes",
    "Oscilloscope, a device displaying visual representations of electronic signals",
    "Overskirt, a garment worn over a skirt, often with ruffles",
    "Bullock cart, a simple cart drawn by oxen",
    "Oxygen mask, a device covering the nose and mouth for breathing assistance",
    "Product packet / packaging, a sealed container for goods",
    "Paddle, a flat, wide implement for rowing or mixing",
    "Paddle wheel, a large wheel with paddles used in watercraft",
    "Padlock, a sturdy lock with a shackle",
    "Paintbrush, a tool with bristles for applying paint",
    "Pajamas, loose-fitting sleepwear often with a matching top and bottom",
    "Palace, a grand building with towers, columns, and intricate details",
    "Pan flute, a musical instrument with pipes of varying lengths",
    "Paper towel, a perforated roll of absorbent paper",
    "Parachute, a fabric device for slowing descent during a fall",
    "Parallel bars, horizontal bars used in gymnastics",
    "Park bench, a long seat in a public outdoor area",
    "Parking meter, a tall device for paying to park",
    "Railroad car, a long vehicle on tracks for transporting goods",
    "Patio, an outdoor space with stone or tile flooring",
    "Payphone, a public telephone with a coin slot",
    "Pedestal, a raised platform or base for displaying objects",
    "Pencil case, a container for holding pencils and pens",
    "Pencil sharpener, a device for sharpening pencil tips",
    "Perfume, a small bottle of scented liquid",
    "Petri dish, a shallow dish used in laboratories",
    "Photocopier, a machine for duplicating documents",
    "Plectrum, a small flat tool for plucking strings",
    "Pickelhaube, a spiked helmet worn by German soldiers",
    "Picket fence, a wooden fence with evenly spaced vertical boards",
    "Pickup truck, a vehicle with an open bed for transporting cargo",
    "Pier, a raised platform extending into the water",
    "Piggy bank, a small container for saving coins",
    "Pill bottle, a cylindrical container for medication",
    "Pillow, a soft cushion for sleeping or resting",
    "Ping-pong ball, a lightweight ball used in table tennis",
    "Pinwheel, a toy with rotating blades in the wind",
    "Pirate ship, a large sailing vessel associated with pirates",
    "Drink pitcher, a container with a handle and spout for pouring liquids",
    "Block plane, a small woodworking tool for shaping wood",
    "Planetarium, a dome-shaped building for displaying celestial projections",
    "Plastic bag, a flexible bag made of synthetic material",
    "Plate rack, a holder for organizing and storing plates",
    "Farm plow, a heavy farming implement for turning soil",
    "Plunger, a rubber suction device used for unclogging drains",
    "Polaroid camera, an instant film camera",
    "Pole, a long, slender cylindrical object",
    "Police van, a vehicle used for transporting police personnel",
    "Poncho, a simple, sleeveless garment worn over clothing",
    "Pool table, a rectangular table with pockets for playing billiards",
    "Soda bottle, a container for carbonated beverages",
    "Plant pot, a container for growing plants",
    "Potter's wheel, a rotating device for shaping clay",
    "Power drill, a handheld tool for drilling holes",
    "Prayer rug, a small carpet used for religious rituals",
    "Printer, a machine for producing copies of documents",
    "Prison, a secure facility for holding individuals",
    "Missile, a guided projectile",
    "Projector, a device for displaying images on a screen",
    "Hockey puck, a flat, hard disk used in ice hockey",
    "Punching bag, a hanging bag used for practicing punches",
    "Purse, a small bag for carrying personal items",
    "Quill, a feather used as a writing instrument",
    "Quilt, a bed covering made of stitched layers of fabric",
    "Race car, a high-speed automobile designed for racing",
    "Racket, a sports implement with a handle and net",
    "Radiator, a device for heating a room",
    "Radio, an electronic device for receiving broadcasts",
    "Radio telescope, a large dish-shaped device for observing radio waves",
    "Rain barrel, a container for collecting rainwater",
    "Recreational vehicle, a motorized vehicle for leisure travel",
    "Fishing casting reel, a device for deploying and retrieving fishing line",
    "Reflex camera, a camera with a mirror for viewing and focusing",
    "Refrigerator, a cooling appliance for storing food",
    "Remote control, a handheld device for operating electronic devices",
    "Restaurant, a place where meals are prepared and served",
    "Revolver, a handgun with a revolving cylinder",
    "Rifle, a long-barreled firearm",
    "Rocking chair, a chair that moves back and forth on curved rockers",
    "Rotisserie, a cooking device for roasting meat",
    "Eraser, a rubber tool for removing pencil marks",
    "Rugby ball, an oval-shaped ball used in rugby",
    "Ruler measuring stick, a straight-edged tool for measuring length",
    "Sneaker, a casual athletic shoe",
    "Safe, a secure box for storing valuables",
    "Safety pin, a fastening device with a clasp and a pointed end",
    "Salt shaker, a container for dispensing salt",
    "Sandal, an open-toed shoe with straps",
    "Sarong, a length of fabric worn as a skirt",
    "Saxophone, a musical instrument with a curved shape and brass body",
    "Scabbard, a sheath for a sword or knife",
    "Weighing scale, a device for measuring weight",
    "School bus, a large vehicle for transporting students",
    "Schooner, a sailing ship with multiple masts",
    "Scoreboard, a display showing points or results in sports",
    "CRT monitor, a bulky monitor with a cathode-ray tube",
    "Screw, a threaded fastener",
    "Screwdriver, a tool for turning screws",
    "Seat belt, a safety restraint worn in vehicles",
    "Sewing machine, a device for stitching fabric",
    "Shield, a protective device with a flat surface",
    "Shoe store, a retail establishment selling footwear",
    "Shoji screen / room divider, a folding screen made of paper or fabric",
    "Shopping basket, a handheld basket for carrying groceries",
    "Shopping cart, a wheeled cart for transporting goods in a store",
    "Shovel, a tool with a broad blade for digging",
    "Shower cap, a waterproof cap worn while showering",
    "Shower curtain, a hanging curtain for enclosing a shower",
    "Ski, a long, narrow snow sports equipment",
    "Balaclava ski mask, a head covering that exposes only the face",
    "Sleeping bag, a portable insulated bag for sleeping outdoors",
    "Slide rule, a flat ruler-like device with sliding parts for making mathematical calculations",
    "Sliding door, a door mounted on a track that opens horizontally by sliding",
    "Slot machine, a gambling device with spinning reels and slots for inserting coins",
    "Snorkel, a tubular device allowing users to breathe while submerged in water",
    "Snowmobile, a vehicle designed for travel on snow, typically with skis at the front",
    "Snowplow, a vehicle equipped with a blade for clearing snow-covered surfaces",
    "Soap dispenser, a device for dispensing liquid soap",
    "Soccer ball, a spherical object used in the sport of soccer",
    "Sock, a knitted or woven garment for the foot",
    "Solar thermal collector, a device that converts sunlight into heat energy",
    "Sombrero, a wide-brimmed hat, often associated with Mexican culture",
    "Soup bowl, a deep dish for serving soup",
    "Keyboard space bar, a large key on a computer keyboard for spacing characters",
    "Space heater, a portable device for heating a small area",
    "Space shuttle, a reusable spacecraft for human spaceflight",
    "Spatula, a flat, broad utensil used for flipping or lifting food",
    "Motorboat, a powered watercraft for recreational or transportation purposes",
    "Spider web, a delicate structure created by spiders for trapping prey",
    "Spindle, a slender rod or pin used for spinning or twisting fibers",
    "Sports car, a high-performance, aerodynamic automobile",
    "Spotlight, a focused beam of light used to illuminate a specific area",
    "Stage, a raised platform for performances or presentations",
    "Steam locomotive, a train powered by a steam engine",
    "Through arch bridge, a bridge with an arch that spans the entire width",
    "Steel drum, a musical instrument made from a metal drum",
    "Stethoscope, a medical instrument used for listening to internal sounds",
    "Scarf, a long piece of cloth worn around the neck or head",
    "Stone wall, a barrier constructed from stones",
    "Stopwatch, a timing device used to measure elapsed time",
    "Stove, a cooking appliance with burners for heating food",
    "Strainer, a perforated utensil for separating liquids from solids",
    "Tram, a vehicle for public transportation on tracks",
    "Stretcher, a portable bed for carrying the sick or injured",
    "Couch, a piece of furniture for seating",
    "Stupa, a Buddhist monument, often with a dome-shaped top",
    "Submarine, a watercraft capable of underwater operation",
    "Suit, a matching set of clothing, typically a jacket and trousers",
    "Sundial, a timekeeping device that uses the sun's position",
    "Sunglasses, protective eyewear against sunlight",
    "Sunglasses, protective eyewear against sunlight",
    "Sunscreen, a lotion for protecting the skin from the sun's rays",
    "Suspension bridge, a bridge supported by cables or chains",
    "Mop, a cleaning tool with absorbent strings or strips",
    "Sweatshirt, a pullover garment for warmth, typically with long sleeves",
    "Swim trunks / shorts, clothing for swimming or casual wear",
    "Swing, a hanging seat suspended from above",
    "Electrical switch, a device for controlling electrical flow",
    "Syringe, a medical instrument for injecting or withdrawing fluids",
    "Table lamp, a light fixture designed for a table",
    "Tank, a large container for holding liquids or gases",
    "Tape player, a device for playing cassette tapes",
    "Teapot, a container with a spout for brewing and serving tea",
    "Teddy bear, a stuffed toy bear",
    "Television, an electronic device for receiving broadcast signals",
    "Tennis ball, a round object used in the sport of tennis",
    "Thatched roof, a roof covering made from dry vegetation",
    "Front curtain, a fabric covering at the front of a stage",
    "Thimble, a small metal or plastic cap worn on the finger for sewing",
    "Threshing machine, a device for separating grains from plants",
    "Throne, an ornate and elevated chair for a person of high status",
    "Tile roof, a roof covering made from ceramic or concrete tiles",
    "Toaster, a kitchen appliance for toasting bread",
    "Tobacco shop, a retail store specializing in tobacco products",
    "Toilet seat, a hinged seat on a toilet bowl",
    "Torch, a portable light source with an open flame",
    "Totem pole, a carved wooden pole with symbolic figures",
    "Tow truck, a vehicle for towing disabled or illegally parked cars",
    "Toy store, a retail store specializing in toys",
    "Tractor, a powerful motor vehicle used for farming",
    "Semi-trailer truck, a large truck with a detachable trailer",
    "Tray, a flat, shallow container for holding or carrying items",
    "Trench coat, a long, waterproof coat with a belt",
    "Tricycle, a three-wheeled vehicle",
    "Trimaran, a boat with three parallel hulls",
    "Tripod, a three-legged stand for supporting a camera or other device",
    "Triumphal arch, a monumental arch commemorating a triumph",
    "Trolleybus, an electric bus powered by overhead wires",
    "Trombone, a brass instrument with a sliding section",
    "Hot tub, a large tub filled with hot water for relaxation",
    "Turnstile, a revolving gate used to control access to a venue",
    "Typewriter keyboard, a set of keys for typing on a typewriter",
    "Umbrella, a folding, canopy-like device for protection against rain",
    "Unicycle, a single-wheeled vehicle",
    "Upright piano, a piano with a vertical soundboard and strings",
    "Vacuum cleaner, a device for cleaning floors by suction",
    "Vase, a decorative container for holding flowers or other items",
    "Vaulted or arched ceiling, a ceiling with a curved or arched shape",
    "Velvet fabric, a soft, luxurious fabric with a dense pile",
    "Vending machine, a machine that dispenses goods when money is inserted",
    "Vestment, a garment worn by clergy during religious ceremonies",
    "Viaduct, a long bridge-like structure for carrying a road or railroad",
    "Violin, a stringed musical instrument played with a bow",
    "Volleyball, a ball used in the sport of volleyball",
    "Waffle iron, a device for cooking waffles",
    "Wall clock, a timekeeping device mounted on a wall",
    "Wallet, a small, flat case for carrying money and cards",
    "Wardrobe, a large, standing closet for storing clothes",
    "Military aircraft, sleek and metallic with powerful engines and wings",
    "Sink, a basin with a faucet for washing hands and dishes",
    "Washing machine, a box-like appliance with a drum for cleaning clothes",
    "Water bottle, a cylindrical container for holding liquid, often made of plastic",
    "Water jug, a larger vessel for holding and dispensing water",
    "Water tower, a tall structure for storing and distributing water",
    "Whiskey jug, a container for holding and pouring whiskey",
    "Whistle, a small, tubular device for making a loud sound",
    "Hair wig, synthetic or natural hair arranged as a head covering",
    "Window screen, a mesh material used to cover windows",
    "Window shade, a fabric covering for a window to control light",
    "Windsor tie, a necktie with a wide, symmetrical knot",
    "Wine bottle, a glass container for holding and serving wine",
    "Airplane wing, the extended horizontal structure on an aircraft",
    "Wok, a round-bottomed cooking pan with high, sloping sides",
    "Wooden spoon, a utensil for stirring or serving, made of wood",
    "Wool, soft and fuzzy fiber from sheep or other animals",
    "Split-rail fence, a fence made of horizontal rails attached to vertical posts",
    "Shipwreck, the remains of a sunken or destroyed ship",
    "Sailboat, a boat with sails for propulsion",
    "Yurt, a circular tent with a wooden frame and felt or fabric covering",
    "Website, a digital platform accessible through the internet",
    "Comic book, a publication with illustrated panels and a narrative",
    "Crossword, a puzzle with intersecting words",
    "Traffic or street sign, a sign providing information or directions",
    "Traffic light, a signal with red, yellow, and green lights for controlling traffic",
    "Dust jacket, a paper cover for a book",
    "Menu, a list of food and drink options at a restaurant",
    "Plate, a flat dish for serving or eating food",
    "Guacamole, a creamy avocado-based dip",
    "Consomme, a clear and clarified broth",
    "Hot pot, a communal cooking pot filled with broth",
    "Trifle, a layered dessert with sponge cake, custard, and fruit",
    "Ice cream, frozen sweetened dairy treat",
    "Popsicle, a frozen dessert on a stick",
    "Baguette, a long, narrow French bread",
    "Bagel, a ring-shaped bread roll",
    "Pretzel, a baked dough snack, typically twisted",
    "Cheeseburger, a sandwich with a ground meat patty and cheese",
    "Hot dog, a cooked sausage in a sliced bun",
    "Mashed potatoes, boiled and mashed potatoes",
    "Cabbage, a leafy green or purple vegetable",
    "Broccoli, a green vegetable with clustered florets",
    "Cauliflower, a white or colored vegetable with compact florets",
    "Zucchini, a green summer squash",
    "Spaghetti squash, a yellow winter squash with stringy flesh",
    "Acorn squash, a small, acorn-shaped winter squash",
    "Butternut squash, a bell-shaped winter squash",
    "Cucumber, a long, cylindrical green vegetable",
    "Artichoke, a thistle-like vegetable with edible leaves",
    "Bell pepper, a hollow, colorful vegetable",
    "Cardoon, a thistle-like plant with edible stalks",
    "Mushroom, a fungus with a cap and stem",
    "Granny Smith apple, a green apple variety",
    "Strawberry, a red, juicy fruit with small seeds on the surface",
    "Orange, a citrus fruit with a bright orange color",
    "Lemon, a sour yellow citrus fruit",
    "Fig, a pear-shaped fruit with sweet flesh",
    "Pineapple, a tropical fruit with a spiky exterior",
    "Banana, a yellow, elongated fruit",
    "Jackfruit, a large tropical fruit with spiky skin and sweet flesh",
    "Cherimoya (custard apple), a heart-shaped fruit with creamy texture",
    "Pomegranate, a round fruit with a tough outer rind and juicy seeds",
    "Hay, dried grass or other plants used for animal bedding or feed",
    "Carbonara, a pasta dish with eggs, cheese, and pancetta",
    "Chocolate syrup, a sweet liquid made from cocoa and sugar",
    "Dough, a mixture of flour, water, and other ingredients for baking",
    "Meatloaf, ground meat mixed with other ingredients and baked",
    "Pizza, a flatbread topped with sauce, cheese, and other toppings",
    "Pot pie, a savory pie with a top crust and filling",
    "Burrito, a rolled tortilla filled with meat, beans, and other ingredients",
    "Red wine, a deep-colored alcoholic beverage made from fermented grapes",
    "Espresso, a strong, concentrated coffee",
    "Tea cup, a small cup for drinking tea",
    "Eggnog, a creamy beverage made with eggs, milk, and sugar",
    "Mountain, a large landform with a peak",
    "Bubble, a small, round sphere of air or gas",
    "Cliff, a steep rock face",
    "Coral reef, a structure formed by coral colonies in the ocean",
    "Geyser, a hot spring that periodically ejects water and steam",
    "Lakeshore, the area along the edge of a lake",
    "Promontory, a high point of land extending into a body of water",
    "Sandbar, a submerged or partially exposed sandbank",
    "Beach, a sandy shore along a body of water",
    "Valley, a low area between hills or mountains",
    "Volcano, a mountain with a vent for lava and gases",
    "Baseball player, an athlete in a baseball uniform",
    "Bridegroom, a man on his wedding day",
    "Scuba diver, a person diving underwater with a breathing apparatus",
    "Rapeseed, a bright yellow flowering plant cultivated for oil",
    "Daisy, a small white or yellow flower with a yellow center",
    "Yellow lady's slipper, an orchid with a yellow pouch-shaped flower",
    "Corn, a cereal plant with large, edible grains",
    "Acorn, the nut of an oak tree",
    "Rose hip, the fruit of a rose plant",
    "Horse chestnut seed, the seed of the horse chestnut tree",
    "Coral fungus, a branching and colorful fungus",
    "Agaric, a type of mushroom with a cap and stem",
    "Gyromitra, a mushroom with a brain-like appearance",
    "Stinkhorn mushroom, a foul-smelling mushroom",
    "Earth star fungus, a fungus with a star-shaped structure",
    "Hen of the woods mushroom, a large, clustered mushroom",
    "Bolete, a mushroom with a cap and porous underside",
    "Corn cob, the central core of an ear of corn",
    "Toilet paper, a roll of paper used for personal hygiene",
    "cyprinid, a slender fish with scales and fins",
    "ray, a flat fish with a broad, diamond-shaped body",
    "thrush, a small bird with speckled plumage and a melodious song",
    "corvine bird, a black bird with a strong beak and glossy feathers",
    "newt, a small amphibian with smooth skin and a long tail",
    "ambystomid, a salamander with a robust body and moist skin",
    "sea turtle, a large marine reptile with a streamlined shell and flippers",
    "iguanid, a lizard with elongated body, scales, and a dewlap",
    "agamid, a lizard with a spiny appearance and a long tail",
    "boa, a large snake with a thick body and smooth scales",
    "rattlesnake, a venomous snake with a rattle at the end of its tail",
    "coraciiform bird, a brightly colored bird with a strong, pointed beak",
    "piciform bird, a bird with a stout bill and zygodactyl feet",
    "duck, a waterfowl with a broad bill, webbed feet, and waterproof feathers",
    "monotreme, a mammal that lays eggs and has a duck-bill and webbed feet",
    "anthozoan, a marine invertebrate with a flower-like appearance",
    "worm, a long, slender invertebrate with a soft body",
    "lobster, a marine crustacean with a hard exoskeleton and claws",
    "stork, a large wading bird with long legs and a straight bill",
    "sandpiper, a small bird with a long bill, slender legs, and a mottled appearance",
    "toy spaniel, a small dog breed with a silky coat and a friendly expression",
    "foxhound, a medium to large dog breed with a muscular build",
    "wolfhound, a large dog breed with a shaggy coat and a strong build",
    "greyhound, a slender dog breed known for its speed and streamlined body",
    "bullterrier, a muscular dog breed with a distinctive egg-shaped head",
    "wirehair, a dog with a wiry coat and distinctive facial features",
    "pointer, a medium to large dog breed with a lean body and a pointed nose",
    "springer spaniel, a medium-sized dog with a wavy coat and long ears",
    "Belgian Sheepdog, a herding dog with a long, black coat and a sturdy build",
    "sled dog, a dog breed adapted for pulling sleds in cold climates",
    "corgi, a small herding dog with a long body, short legs, and pointed ears",
    "wildcat, a small to medium-sized feline with a sleek coat and pointed ears",
    "viverrine, a carnivorous mammal with a slender body and a spotted coat",
    "scarabaeid beetle, a beetle with a hard, shell-like exoskeleton and distinctive antennae",
    "hymenopterous insect, an insect with membranous wings and often a stinger",
    "orthopterous insect, an insect with strong hind legs adapted for jumping",
    "dictyopterous insect, an insect with leathery wings and a flattened body",
    "homopterous insect, an insect with piercing-sucking mouthparts and often a waxy coating",
    "odonate, a flying insect with transparent wings, elongated body, and large eyes",
    "rabbit, a small mammal with long ears and a fluffy tail",
    "equine, a mammal belonging to the horse family",
    "edentate, a mammal without teeth, such as an anteater or sloth",
    "lesser ape, a small to medium-sized primate with a tail",
    "lemur, a primate with large eyes and a long, bushy tail",
    "elephant, a large mammal with a long trunk and tusks",
    "procyonid, a mammal in the raccoon family with a ringed tail",
    "ganoid, a fish with thick, rhomboid scales",
    "robe, a long, flowing garment typically worn for formal occasions",
    "gown, a woman's dress, typically formal or elegant",
    "guitar, a musical instrument with a wooden body and strings",
    "Warship, a formidable vessel with heavy armaments and a sleek, imposing structure",
    "Lighter-than-air craft, a floating vehicle filled with gas, typically characterized by its large, balloon-like shape",
    "Shed, a simple structure with a sloping roof, often used for storage or shelter",
    "Protective garment, an article of clothing designed to shield the body from harm or danger",
    "Weight, a measure of heaviness or mass, often represented by a hanging scale",
    "Handcart, a wheeled conveyance for transporting goods, featuring a simple frame and handles",
    "Double-reed instrument, a musical device producing sound through two vibrating reeds, like the oboe",
    "Towel, a fabric used for drying or wiping, typically rectangular and absorbent",
    "Jar, a container with a narrow neck and a wide mouth, suitable for holding liquids or preserved items",
    "Glass, a transparent material with a smooth surface, commonly used for containers or windows",
    "Bicycle, a two-wheeled vehicle propelled by pedaling, with a frame and handlebars",
    "Sled, a sliding vehicle designed for gliding over snow or ice, often with runners",
    "Cap, a head-covering accessory, usually with a brim or visor",
    "Memorial, a structure or object created to honor and remember a person or event",
    "Armor plate, a protective layer made of sturdy material, worn for defense in battle",
    "Cleaning implement, a tool or device used to remove dirt or impurities",
    "Small boat, a compact watercraft suitable for personal use, often powered by oars or a small motor",
    "Opener, a tool for removing caps or lids from containers",
    "Sweater, a knitted garment worn on the upper body, typically with long sleeves",
    "Sailboat, a boat propelled by sails, featuring a mast and rigging",
    "Bowed stringed instrument, a musical device with strings played by a bow, such as a violin",
    "Body armor, protective gear worn on the body for defense against impact or projectiles",
    "Cabinet, a piece of furniture with shelves or drawers, used for storage or display",
    "Theater, a building or space designed for the performance of plays or other dramatic works",
    "Knife, a sharp-edged cutting tool with a handle",
    "Dwelling, a place of residence or habitation, often a house or apartment",
    "Shaker, a container with a perforated top for sprinkling or shaking",
    "Lock, a mechanical device for securing doors or containers",
    "Keyboard, a set of keys or buttons, typically on a musical instrument or computer",
    "Helmet, a protective headgear, often used in sports or combat",
    "Staff, a long, slender rod or stick, frequently used for support or as a symbol",
    "Table, a flat surface supported by legs, used for various activities",
    "Oven, a cooking appliance with a chamber for baking or roasting",
    "Locomotive, a powered railway vehicle for pulling trains",
    "Makeup, substances applied to enhance facial appearance",
    "Scarf, a length of fabric worn around the neck for warmth or fashion",
    "Car, a motorized vehicle for transportation on roads",
    "Pan, a shallow, flat-bottomed container for cooking",
    "Mask, a covering for the face, often worn for protection or disguise",
    "piano, a musical instrument with black and white keys",
    "basket, a container woven from natural materials",
    "farm machine, a mechanical device used in agriculture",
    "sheath, a protective covering or case",
    "framework, a structure composed of interconnected parts",
    "puzzle, an entertainment consisting of pieces to be fitted together",
    "control, a device for managing or directing a system",
    "cap, a head covering with a visor",
    "electro-acoustic transducer, a device converting electrical signals into sound",
    "bowl, a round, deep dish for holding food or liquids",
    "electronic device, a gadget powered by electricity",
    "trap, a device for catching or restraining",
    "column, a vertical pillar or post",
    "filter, a device for removing impurities or particles",
    "breathing device, a tool facilitating respiration",
    "residence, a place where someone lives",
    "timer, a device measuring or indicating time intervals",
    "camera, a device capturing images or recording videos",
    "communication system, a network for exchanging information",
    "curtain, a piece of fabric hanging to cover or decorate a window",
    "slot machine, a gambling device with spinning symbols",
    "reflector, a surface redirecting light or sound waves",
    "heater, a device producing warmth",
    "medical instrument, a tool used in healthcare practices",
    "fabric, material produced by weaving or knitting fibers",
    "heavier-than-air craft, an aircraft that stays aloft due to lift",
    "jug, a container with a narrow neck and handle",
    "frozen dessert, a sweet, cold dish frozen before consumption",
    "loaf of bread, a shaped mass of baked flour and water",
    "sandwich, a food item with ingredients between two slices of bread",
    "summer squash, a tender, edible gourd harvested in the summer",
    "winter squash, a hard-skinned gourd harvested in the winter",
    "citrus, a type of fruit from the citrus genus",
    "sauce, a flavorful liquid or semi-solid condiment",
    "punch, a drink made with fruit juices and often spiked",
    "mountain, a large elevated landform",
    "shore, the land along the edge of a body of water",
    "flower, a colorful and fragrant reproductive structure in plants",
    "whale, a large marine mammal",
    "ridge, a long, elevated crest",
    "percoid fish, a type of fish belonging to the perch family",
    "Crocodilian reptile, a scaly, aquatic creature with a long snout and powerful tail",
    "Elapid, a venomous snake with a slender body and fixed, hollow fangs",
    "Magnifier, a small device with a convex lens for enlarging objects",
    "Movable barrier, a partition that can be shifted or repositioned easily",
    "Floor cover, a flat surface covering for the ground, often made of various materials",
    "Gear, mechanical equipment consisting of interlocking cogs or wheels",
    "Cloth covering, a fabric layer used to conceal or protect",
    "Shark, a large predatory fish with sharp teeth and streamlined body",
    "Frog, a small amphibian with smooth, moist skin and powerful hind legs",
    "Viper, a venomous snake with a triangular-shaped head and fangs",
    "Phasianid, a bird of the pheasant family, often with vibrant plumage",
    "Anseriform bird, a waterfowl with a broad, flat bill and webbed feet",
    "Marsupial, a mammal with a pouch for carrying and nurturing young",
    "Coelenterate, a marine invertebrate with tentacles and a gelatinous body",
    "Heron, a long-legged wading bird with a slender neck and bill",
    "Schnauzer, a dog breed with distinctive facial hair and a robust build",
    "Setter, a hunting dog with a long coat and a distinctive stance",
    "Pinscher, a compact dog breed with a short coat and erect ears",
    "Poodle, a curly-coated dog breed with a distinctive haircut",
    "Wild dog, a carnivorous mammal in its natural, untamed state",
    "Echinoderm, a marine invertebrate with a spiny skin, like a starfish",
    "Leporid, a small mammal resembling a rabbit, with long ears and hind legs",
    "Swine, a domesticated pig or hog with a stout body and snout",
    "Antelope, a graceful herbivorous mammal with curved horns",
    "Great ape, a large, intelligent primate with no tail, like a chimpanzee",
    "Clock, a timekeeping device with hands or digits on a round face",
    "Gymnastic apparatus, equipment used in gymnastics, such as bars or beams",
    "Pen, a writing instrument with ink and a pointed nib",
    "Baby bed, a small, protective sleeping space for infants",
    "Cap, a headcovering with a brim or visor",
    "Shelter, a structure providing protection or refuge from the elements",
    "Swimsuit, a form-fitting garment worn for swimming",
    "Optical instrument, a tool for observing or measuring using lenses or mirrors",
    "Necktie, a long piece of cloth worn around the neck for formal occasions",
    "Pot, a deep, rounded container for cooking or holding liquids",
    "Lamp, a source of light with a bulb or flame",
    "Place of worship, a building or site dedicated to religious practices",
    "Brass, a yellowish metal alloy, often used for decorative purposes",
    "Telephone, a communication device with a receiver and dial or buttons",
    "White goods, large household appliances like refrigerators and washing machines",
    "Screen, a flat surface that displays images or information",
    "Coat, an outer garment worn for warmth, typically with sleeves and buttons",
    "Portable computer, a small, lightweight electronic device for computing on the go",
    "Cart, a four-wheeled vehicle used for transporting goods",
    "Shoe, a covering for the foot, typically made of leather or other material",
    "Top, a garment covering the upper part of the body",
    "System, a set of connected components forming a complex whole",
    "Bus, a large motor vehicle designed to carry passengers",
    "Van, a medium to large-sized vehicle for transporting goods or people",
    "Dish, a shallow container for serving or cooking food",
    "Support, a structure or device that bears the weight of something",
    "Firearm, a portable gun, typically with a barrel and bullets",
    "Sailing vessel, a watercraft propelled by sails",
    "Signboard, a sign with information or advertising",
    "Armor, protective covering worn to prevent injury",
    "Hosiery, stockings, socks, or other garments worn on the feet and legs",
    "Bridge, a structure built to span a physical obstacle",
    "Cruciferous vegetable, a vegetable of the cabbage family, often with four-petaled flowers",
    "Seed, a mature fertilized ovule containing a young plant",
    "Beating-reed instrument, a musical instrument with a reed that vibrates",
    "Edge tool, a cutting or shaping tool with a sharp edge",
    "Seabird, a bird that lives and feeds near the sea",
    "Supporting structure, a framework that holds something in place",
    "Blind, a covering for a window to block light",
    "Dessert, a sweet course typically served at the end of a meal",
    "Alcohol, a colorless volatile liquid used in drinks",
    "Condiment, a substance used to add flavor to food",
    "Person, a human being",
    "Bird of prey, a carnivorous bird that hunts and feeds on other animals",
    "Grouse, a game bird with a plump body and feathered legs",
    "Parrot, a colorful, long-tailed bird often capable of mimicking human speech",
    "Gastropod, a mollusk with a single spiral shell",
    "Crab, a crustacean with a broad, flat body and pincers",
    "Sennenhunde, a large Swiss mountain dog breed",
    "Spitz, a dog breed with a thick double coat and pointed ears",
    "Wolf, a wild carnivorous mammal",
    "Fox, a small carnivorous mammal with a bushy tail",
    "Bear, a large, powerful mammal with thick fur",
    "Soft-finned fish, a type of fish with fins supported by soft rays",
    "Chair, a seat for one person, typically with a back and four legs",
    "Hat, a covering for the head",
    "Gun, a portable firearm",
    "Wheel, a circular object that rotates on an axle",
    "Fence, a barrier enclosing an area",
    "Personal computer, a computer for individual use",
    "Roof, the upper covering of a building",
    "Woodwind, a musical instrument with a wooden body and a reed",
    "Skirt, a garment worn around the waist, typically by women",
    "aquatic mammal, a marine creature with a streamlined body and flippers",
    "aircraft, a flying machine with wings and engines",
    "spiny-finned fish, a fish with sharp, pointed fins along its body",
    "source of illumination, an object emitting light",
    "squash, a vegetable with a round shape and smooth skin",
    "course, a designated path or route",
    "overgarment, an outer piece of clothing worn over other garments",
    "bread, a baked food item with a brown crust",
    "public transport, a vehicle for carrying passengers in public",
    "finch, a small bird with a pointed beak and colorful plumage",
    "salamander, a lizard-like amphibian with smooth, moist skin",
    "turtle, a reptile with a hard, protective shell",
    "shorebird, a bird found along the water's edge",
    "retriever, a dog breed with a water-resistant coat and retrieving skills",
    "watchdog, a vigilant dog trained to guard and protect",
    "domestic cat, a small, furry pet with retractable claws",
    "bag, a container with an opening at the top",
    "piece of cloth, a fabric material with various textures and colors",
    "footwear, items worn on the feet for protection and comfort",
    "stick, a long, slender piece of wood or other material",
    "natural elevation, a raised area of land",
    "anthropoid ape, a primate with a large body and no tail",
    "elasmobranch, a cartilaginous fish with a flattened body and sharp teeth",
    "spider, an arachnid with eight legs and silk-spinning abilities",
    "mollusk, a soft-bodied invertebrate with a protective shell",
    "big cat, a large feline species, such as a lion or tiger",
    "butterfly, an insect with colorful wings and a slender body",
    "rodent, a small mammal with continuously growing incisor teeth",
    "New World monkey, a primate found in the Americas with a prehensile tail",
    "weapon, an object designed for inflicting harm or damage",
    "box, a container with a flat base and sides",
    "percussion instrument, a musical instrument played by striking",
    "boat, a watercraft designed for travel on water",
    "toiletry, personal care items used for grooming",
    "kitchen appliance, a device used for food preparation in the kitchen",
    "ship, a large watercraft for transporting goods or people",
    "computer, an electronic device for processing and storing information",
    "spaniel, a dog breed with long ears and a silky coat",
    "musteline mammal, a carnivorous mammal, such as a weasel or ferret",
    "Old World monkey, a primate found in Africa and Asia",
    "bottle, a container with a narrow neck for holding liquids",
    "ball, a round object used in various sports and games",
    "hand tool, a manually operated tool for specific tasks",
    "seat, a furniture item for sitting",
    "dish, a shallow container for serving or eating food",
    "fungus, a group of organisms with diverse shapes and colors",
    "cat, a small domesticated carnivore with retractable claws",
    "game bird, a bird species hunted for sport or food",
    "cooking utensil, tools used in the kitchen for preparing food",
    "decapod crustacean, a crustacean with ten limbs, such as a crab or lobster",
    "toy dog, a small breed of dog kept as a companion",
    "beetle, an insect with a hard outer shell and distinctive wing covers",
    "shop, a place where goods or services are sold",
    "stringed instrument, a musical instrument with strings that produce sound",
    "fastener, a device used to join or secure objects",
    "truck, a motor vehicle designed for transporting goods",
    "timepiece, a device for measuring and displaying time",
    "amphibian, a cold-blooded vertebrate that can live in water and on land",
    "Sports equipment, various gear for athletic activities",
    "Foodstuff, consumable items for nourishment",
    "Colubrid snake, a slender serpent with smooth scales",
    "Arachnid, eight-legged creature with a segmented body",
    "Bovid, a mammal with cloven hooves and usually curved horns",
    "Car, four-wheeled motor vehicle for transportation",
    "Mechanical device, a contraption with moving parts for specific functions",
    "Game equipment, tools used in various sports and games",
    "Crustacean, marine or freshwater arthropod with a hard exoskeleton",
    "Mercantile establishment, a place for buying and selling goods",
    "Headdress, an ornament worn on the head for decoration",
    "Barrier, a structure that blocks or hinders movement",
    "Electronic equipment, devices powered by electricity for various purposes",
    "Edible fruit, a tasty and nutritious reproductive body of a flowering plant",
    "Geological formation, natural features shaped by geological processes",
    "Teleost fish, a bony fish with a protruding jaw",
    "Establishment, an organization or place of business",
    "Oscine, a songbird known for melodious vocalizations",
    "Home appliance, electrical devices for household tasks",
    "Restraint, a device or measure that limits movement",
    "Wind instrument, a musical device played by blowing air",
    "Lizard, a reptile with scaly skin and usually four legs",
    "Nutriment, substances that provide nourishment",
    "Shepherd dog, a canine bred for herding livestock",
    "Measuring instrument, a tool for determining dimensions or quantities",
    "Mechanism, a system of parts working together for a specific purpose",
    "Vegetable, a plant or part of a plant consumed as food",
    "Feline, a carnivorous mammal, often domesticated as a pet",
    "Monkey, a primate with a tail, opposable thumbs, and often agile movements",
    "Machine, a device with moving parts that performs a task",
    "Building, a structure with walls and a roof",
    "Tool, a handheld device used to perform a specific task",
    "Even-toed ungulate, a hoofed mammal with an even number of toes",
    "Vessel, a container for holding liquids or other substances",
    "Wading bird, a bird that frequents shallow waters",
    "Fruit, the mature ovary of a flowering plant, typically sweet or savory",
    "Fish, aquatic vertebrates with gills and fins",
    "Snake, a legless reptile with elongated body and no eyelids",
    "Sporting dog, a breed of dog trained for hunting or retrieving",
    "Ungulate, a mammal with hooves",
    "Hound, a type of hunting dog with a keen sense of smell",
    "Vessel, a watercraft designed for transportation",
    "Primate, a group of mammals including humans, apes, and monkeys",
    "Craft, a skill in making things by hand",
    "Motor vehicle, a self-propelled conveyance for transportation",
    "Furniture, movable items used to support human activities",
    "Food, items consumed for sustenance",
    "Aquatic bird, a bird adapted for life on or near water",
    "Garment, an item of clothing",
    "Substance, a particular kind of matter with uniform properties",
    "terrier, a small, wiry dog with a sturdy build and alert expression",
    "protective covering, a shield-like layer providing defense or shelter",
    "musical instrument, an object producing sound through vibration or resonance",
    "insect, a small, six-legged creature with a distinct segmented body",
    "instrument, a tool designed for a specific purpose, often with precision",
    "working dog, a diligent canine trained to assist in various tasks",
    "self-propelled vehicle, a mobile machine capable of moving on its own",
    "plant, a living organism with leaves, stems, and roots for photosynthesis",
    "diapsid, a reptile characterized by two openings in the skull",
    "equipment, tools or devices used for a particular activity or purpose",
    "reptile, a cold-blooded vertebrate with scales, often laying eggs",
    "implement, a tool or utensil designed for practical use",
    "wheeled vehicle, a conveyance equipped with wheels for transportation",
    "arthropod, an invertebrate with an exoskeleton and jointed legs",
    "clothing, items worn on the body for protection or fashion",
    "container, an object designed to hold and transport items",
    "bird, a feathered, warm-blooded vertebrate with wings for flight",
    "structure, a constructed arrangement of parts forming a whole",
    "invertebrate, an animal lacking a backbone or spinal column",
    "hunting dog, a canine trained for tracking and pursuing game",
    "vehicle, a conveyance designed for transporting people or goods",
    "conveyance, a means of transporting or carrying something",
    "covering, a layer or material used to conceal or protect",
    "dog, a domesticated carnivorous mammal, often kept as a pet",
    "device, a mechanical or electronic tool for a specific purpose",
    "canine, a member of the dog family, known for loyalty and companionship",
    "carnivore, an animal primarily consuming meat for sustenance",
    "placental, a mammal giving birth to live offspring nourished by a placenta",
    "mammal, a warm-blooded vertebrate with hair or fur and mammary glands",
    "vertebrate, an animal with a backbone or spinal column",
    "instrumentality, a tool or means for accomplishing a specific purpose",
    "animal, a living organism with sensory and locomotive abilities",
    "organism, a living being capable of growth, reproduction, and response",
    "artifact, a human-made object or item",
    "physical entity, a tangible and perceptible substance or object"]